/*
 * SOURCE:  intui.c
 * PROJECT: EasyTeX
 *
 * PURPOSE: windows, menus, requester...
 *
 * UPDATES: 08/15/1991 - major rewrite
 *          10/21/1991 - CUSTOMGADGET added
 *          03/17/1992 - default gadget types
 *          04/27/1992 - bug in _msgbox fixed
 *          06/13/1992 - bug in openrequest fixed
 *
 * (c)M.Schollmeyer
 */
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>

#include "intui.h"

static struct _window {
    int NewLeft, NewTop;      /* left and top offset */
    int Width, Height;  /* width and height of window */
    int Left, Top;
    char *Title;
    unsigned long Flags;      /* window flags s.a. */
    void *Buffer;             /* buffer to backup screen frame */
};

static struct _request {
    struct IntuiText *Title;
    struct IntuiText **Items;
    unsigned long Flags;
    int RetVal;
    int CurrIt;
    struct Gadget *TextLink;
    int BoxTop;
    int NumIt;
};

static struct _intuitionbase {
    struct InputEvent *ie;
    unsigned int mouse_status;
    unsigned int VideoBase;
    colorreg GadgetColor;
    colorreg GadgetSelect;
    colorreg GadgetHot;
    colorreg GadgetHotSelect;
    colorreg WindowColor;
    colorreg BorderColor;
    colorreg TextColor;
    int soundfreq;
    int sounddur;
};

static struct _intuitionbase ibase = {
    NULL, 0, 0, CR_BOOLGADGET, CR_BOOLGADSELECT, CR_GADHOTKEY,
    CR_GADSELHOT, CR_WINDOW, CR_BORDER, CR_WINDOWTEXT };


static void _drawborder( int, int, int, int, unsigned long, colorreg, int, int );
static void _printchar( unsigned char, int, colorreg, int, int );

static void _printl( char *, int, colorreg, int, int );
static void _nprint( char *, int, colorreg, int, int );

static struct Gadget *_parentgadget( struct Gadget *, struct Gadget * );
static struct Gadget *_lastgadget( struct Gadget * );
static struct Gadget *_firstgadget( struct Gadget * );
static struct Gadget *_findkeygadget( struct Gadget *, unsigned int );
static void _drawgadget( struct Gadget *, int, int, unsigned int );
static int _drawgadstring( char *, int, int, colorreg, colorreg );

static struct InputEvent *_editboolgadget( struct Gadget *, struct Gadget *, int, int );
static struct InputEvent *_editcheckgadget( struct Gadget *, struct Gadget *, int, int );
static struct InputEvent *_edittextgadget( struct Gadget * , struct Gadget *, int, int );

static struct InputEvent *_editpopgadget( struct Gadget *, struct Gadget *, int, int );

struct MenuItem *_finditem( struct MenuItem *, int , int );

static struct InputEvent *_openrequest( struct Gadget *, struct Gadget *, int, int );
static void _drawrequest( struct Gadget *, int, int );

static char _altname( unsigned int );


struct IntuitionBase *OpenIntuition( int version )
{

    if( version > 1 )
    {
        return( (struct IntuitionBase *)0L );
    }

    SetDoubleClick( DOUBLECLICKSPEED );


    ibase.ie = InputEventAddr();
    ibase.ie->mousestatus = ResetMouse();

    ibase.soundfreq = 780;
    ibase.sounddur = 1;

    return( (struct IntuitionBase *)&ibase );
}

/*
 *      NAME:           DrawBorder
 *      ARGUMENTS:      
 *      RETURN TYPE:    
 *
 *      Purpose:        
 *
 * (c) M. Schollmeyer 1990,1991
 */
void DrawBorder( struct Border *b, int left_off, int top_off )
{
    for( ; b; b = b->Next )
    {
        _drawborder( b->Left, b->Top, b->Width, b->Height
            , b->Flags, ibase.BorderColor, left_off, top_off );

        if( b->Title )
            VideoStr( ibase.BorderColor, b->Title,
                b->Left+(b->Width-strlen(b->Title))/2+left_off, b->Top+top_off );
    }
}


static void _drawborder( left, top, width, height, flags, color, left_off, top_off )
int left, top, width, height;
unsigned long flags;
colorreg color;
int left_off, top_off;
{
    int i;
    char *c, *bars;
    static unsigned char twice[] = "ɻȼ";
    static unsigned char doublebars[] = "ͺ";
    static unsigned char *single[] = {
        "ڿ" ,"" ,"ڴ" ,"¿" ,"ڿ" ,""     ,
        "ڴ" ,""     ,"ÿ" ,"" ,""     ,""     ,
        "ÿ" ,        };


    static unsigned char *singlebars = "ĳ";

    if( flags & DOUBLE_BORDER )
    {
        c = twice;
        bars = doublebars;
    }
    else
    {
        c = single[ flags >> 28 ];
        bars = singlebars;
    }

    VideoOut( color, c[0], left + left_off, top + top_off );
    _printchar( bars[0], width - 2, color, left + left_off + 1, top + top_off);
    VideoOut( color, c[1], left + width + left_off - 1, top + top_off );
    for( i = 1; i < height - 1; ++i)
    {
        VideoOut( color, bars[1], left + left_off, i + top + top_off);
        VideoOut( color, bars[1], left + width + left_off - 1, i + top + top_off );
    }
    VideoOut( color, c[2], left + left_off, top + height + top_off- 1 );
    _printchar( bars[0], width - 2, color, left + left_off + 1, top + top_off + height - 1);
    VideoOut( color, c[3], left + left_off + width - 1, top + top_off + height - 1 );

}


static void _printchar( unsigned char c, int number, colorreg color, int col, int row )
{
    while( number-- > 0 )
    {
        VideoOut( color, c, col, row );
        ++col;
    }
}


/*
 *      NAME:           CenterText
 *      ARGUMENTS:      
 *      RETURN TYPE:    
 *
 *      Purpose:        
 *
 * (c) M. Schollmeyer 1990,1991
 */
void CenterText( struct IntuiText *it, int width ) {

    it->Left = ( width - strlen( it->Text ) ) / 2;
}

/*
 *      NAME:           PrintIText
 *      ARGUMENTS:      
 *      RETURN TYPE:    
 *
 *      Purpose:        
 *
 * (c) M. Schollmeyer 1990,1991
 */
int PrintIText( struct IntuiText *t, int left_off, int top_off )
{
    unsigned int len, width, line;
    char *cp, *end;

    HideMouse();

    for( ; t != 0L; t = t->Next)
    {
        len = strlen( t->Text );
        if( t->Flags & IT_VERTICAL )
            width = 1;
        else if( t->Width == 0 )
            width = _MAX( len, 1 );
        else
            width = t->Width;

        if( t->TextBorder )
        {
            _drawborder( t->TextBorder->Left,
                         t->TextBorder->Top,
                         t->TextBorder->Width,
                         t->TextBorder->Height,
                         t->TextBorder->Flags,
                         ibase.WindowColor,
                         t->Left,
                         t->Top );
        }

        cp = t->Text;
        end = cp + len;
        for( line = 0; cp < end; ++line, cp += width )
            _printl( cp, width, ibase.TextColor,
                t->Left + left_off, t->Top + top_off + line );
    }

    ShowMouse();
    return line;
}



static void _printl( char *cp, int len, colorreg color, int col, int row )
{
    for( ;  *cp && len ; ++cp, ++col, --len )
    {
        VideoOut( color, *cp, col, row );
    }
}

static void _nprint( char *cp, int len, colorreg color, int col, int row ) {

    for( ; *cp && len; ++cp, ++col, --len ) {
        VideoOut( color, *cp, col, row );
    }
    for( ; len; ++cp, ++col, --len ) {
        VideoOut( color, ' ', col, row );
    }
}


/*
 *      NAME:           ClearScreen
 *      ARGUMENTS:      
 *      RETURN TYPE:    
 *
 *      Purpose:        
 *
 * (c) M. Schollmeyer 1990,1991
 */
void ClearScreen( void )
{
    unsigned char w, h;

    w = VideoBase->ScreenWidth;
    h = VideoBase->ScreenHeight;

    _asm {
        mov     bh, CR_DOS
        xor     cx, cx
        mov     dh, h
        mov     dl, w
        xor     ax, ax
        mov     ah, 06h
        int     10h
    }
}

/*
 *      NAME:           OpenWindow
 *      ARGUMENTS:      
 *      RETURN TYPE:    
 *
 *      Purpose:        
 *
 * (c) M. Schollmeyer 1990,1991
 */
// #pragma optimize( "elg", off )
int OpenWindow( struct Window *window, int topic )
{
    struct _window *w;
    int _far *buf;
    int i, j;
    int isborder = 1;

    w = (struct _window *)window;

    w->Left = w->NewLeft;
    if( w->Left == -1 )
        w->Left = (VideoBase->ScreenWidth-w->Width)/2;
    w->Top = w->NewTop;
    if( w->Top == -1 )
        w->Top = (VideoBase->ScreenHeight-w->Height)/2;

    if( !( w->Flags & STATIC_WINDOW ) )
    {
        w->Buffer = AllocHeap( (w->Width+2) * (w->Height+1) * 2 );
        if( w->Buffer == NULL )
        {
            return FALSE;
        }
    }
    buf = ( int _far * )w->Buffer;

    HideMouse();
    for( i = w->Left ; i < w->Left + w->Width + 2; ++i )
    {
        for( j = w->Top; j < w->Top + w->Height + 1; ++j, ++buf )
        {
            *buf = VideoGet( i, j );
        }
    }

    RefreshWindow( window );
    ShowMouse();

    PushTopic( topic );

    return TRUE;
}
// #pragma optimize( "", on )

/*
 *      NAME:           RefreshWindow
 *      ARGUMENTS:      
 *      RETURN TYPE:    
 *
 *      Purpose:        
 *
 * (c) M. Schollmeyer 1990,1991
 */
int RefreshWindow( struct Window *window )
{
    struct _window *w;
    int i, j;
    int isborder = 1;

    w = (struct _window *)window;


    if( w->Flags & BORDERLESS )
            isborder = 0;
    else
        _drawborder( w->Left, w->Top, w->Width, w->Height, 0L, ibase.WindowColor, 0,0 );
        
    for( i = isborder; i < w->Height - isborder; ++i)
    {
        _printchar( ' ', w->Width - 2 * isborder, ibase.WindowColor,
                 w->Left + isborder, i + w->Top );
    }

    if( !(w->Flags & SHADOWLESS ) ) {
         for( i = w->Left, j = w->Top + w->Height;
             i < w->Left + w->Width; ++i )
             VideoAttr( CR_SHADOW, i + 2, j );
         for( i = w->Left + w->Width, j = w->Top + 1;
             j < w->Top + w->Height; ++j ) {
             VideoAttr( CR_SHADOW, i , j );
             VideoAttr( CR_SHADOW, i + 1, j );
        }
    }

    if( w->Title )
        VideoStr( ibase.WindowColor, w->Title,
            w->Left+(w->Width-strlen(w->Title))/2, w->Top );

    return TRUE;
}

/*
 *      NAME:           CloseWindow
 *      ARGUMENTS:      
 *      RETURN TYPE:    
 *
 *      Purpose:        
 *
 * (c) M. Schollmeyer 1990,1991
 */
void CloseWindow( struct Window *window )
{
    struct _window *w;
    int _far *buf;
    int i, j;

    w = (struct _window *)window;

    PopTopic();

    buf = ( int _far * )w->Buffer;

    HideMouse();
    for( i = w->Left ; i < w->Left + w->Width + 2; ++i )
    {
        for( j = w->Top; j < w->Top + w->Height + 1; ++j, ++buf )
        {
            VideoPut( *buf, i, j );
        }
    }
    ShowMouse();
    if( !( w->Flags & STATIC_WINDOW ) )
    {
        free( w->Buffer );
    }
}

/*
 *      NAME:           OpenGadget
 *      ARGUMENTS:      struct Gadget * :   Pointer to first Gadget
 *                      int left_off    :   Left Offset of all Gadgets
 *                      int top_off     :   Top Offset
 *      RETURN TYPE:    struct Gadget * :   Pointer to last active Gadget
 *
 *      Purpose: OpenGadget opens a linked list of gadgets and queries
 *               use input. It then returns a pointer to the last active
 *               gadget or NULL if <ESC> was pressed.
 *
 * OpenGadget's structure looks like this:
 *
 *  OpenGadget()
 *  {
 *      FOR EACH gadget IN GADGETLIST
 *          drawgadget
 *
 *      <if any gadget is ACTIVATED, it is displayed like this and
 *       'curr' points to that gadget, otherwise 'curr' is NULL>
 *
 *      FOR EVER <main message loop>
 *          IF curr == NULL
 *              ReadInput
 *              <handle only keyinput, e.g. esc and return>
 *          ELSE
 *              <handle gadget, one routine for each gadget type>
 *
 *          IF keyinput
 *              <handle key input>
 *          ELSE IF mouseinput
 *              <handle mouse input>
 *      END
 *
 *
 * (c) M. Schollmeyer 1990,1991
 */
#define _CURRGADGET 0x7f
#define _NOTCURRGAD 0x00

static struct Gadget *( *gadhandler )();

// #pragma optimize( "elg", off )
struct Gadget *OpenGadget( struct Gadget *g, int left_off, int top_off,
    struct Gadget *(*handler)() )
{
    struct Gadget *first, *curr, *new, *gp;
    struct InputEvent *ie;
    struct Cursor sav;
    struct StringInfo _far *si;
    struct _request _far *_r;
    struct IntuiText **it;
    BOOL nowait;
    struct Gadget *(*oldhandler)();

    static struct IntuiText it_ok = {
        NULL, NULL, "OK", 1, 0
    };
    static struct IntuiText it_cancel = {
        NULL, NULL, "Cancel", 1, 0
    };
    static struct IntuiText it_help = {
        NULL, NULL, "&Help", 1, 0
    };

    oldhandler = gadhandler; /* save old function pointer on stack */
    gadhandler = handler;    /* set new handler */

    GetCursor( -1, &sav );
    ClearCursor();

    first = g;
    curr = NULL;

    for( ; g; g = g->Next )
    {
        if( g->Flags & LONGINT )
        {
            si = (struct StringInfo _far *)(g->SpecialInfo);
            ltoa( si->LongInt, si->Buffer, 10 );
        }
        else if( g->Flags & REQGADGET )
        {
            _r = (struct _request _far *)(g->SpecialInfo);
            _r->RetVal = RETURNPENDING;
            if( _r->Flags & RESETREQUEST ) {
                for( it = _r->Items; *it; ++it )
                    CLEARBIT( (*it)->Flags, IT_SELECTED );
                CLEARBIT( _r->Flags, RESETREQUEST );
            }
        }
        else if( g->Flags & DEFOKGADGET )
        {
            g->IText = &it_ok;
            g->Width = 6;
            g->Height = 3;
            g->Flags = DEFOKGADGET|BOOLGADGET|ENDGADGET;
            g->SpecialInfo = NULL;
            g->MutalExclude = 0L;
        }
        else if( g->Flags & DEFCANCELGADGET )
        {
            g->IText = &it_cancel;
            g->Width = 10;
            g->Height = 3;
            g->Flags = DEFCANCELGADGET|BOOLGADGET|ENDGADGET;
            g->SpecialInfo = NULL;
            g->MutalExclude = 0L;
        }
        else if( g->Flags & DEFHELPGADGET )
        {
            g->IText = &it_help;
            g->Width = 8;
            g->Height = 3;
            g->Flags = DEFHELPGADGET|BOOLGADGET|ENDGADGET;
            g->SpecialInfo = NULL;
            g->MutalExclude = 0L;
        }

        if( g->Flags & ACTIVATED )
        {
            curr = g;
        }
        else
        {
            _drawgadget( g , left_off, top_off, _NOTCURRGAD );
        }

        if( gadhandler )
            (*gadhandler)( first, g, left_off, top_off, CG_INITIALIZE );

    }

    if( curr )
    {
        _drawgadget( curr, left_off, top_off, _CURRGADGET );
    }

    for( ;; )
    {
        if( curr == NULL )
        {
            ie = ReadInput();
            if( ie->key )
            {
                if( ie->key == BK_RETURN || ie->key == BK_ESC )
                {
                    goto quit;
                }

                nowait = TRUE;
                new = _findkeygadget( first, ie->key );
                if( new == NULL ) curr = first;
                else goto again;
                _drawgadget( curr, left_off, top_off, _CURRGADGET );
                continue;
            }
            /* if it is a mouse input, fall thru */
        }

        else if( curr->Flags & TEXTGADGET )
        {
            ie = _edittextgadget( curr, first, left_off, top_off );
        }
                
        else if( curr->Flags & BOOLGADGET )
        {
            ie = _editboolgadget( curr, first, left_off, top_off );
        }

        else if( curr->Flags & REQGADGET )
        {
            struct _request _far *_r;

            _r = (struct _request _far *)(curr->SpecialInfo);
            ie = _openrequest( first, curr, left_off, top_off );
            /* _openrequest handles keyboard input like this... */
            switch( _r->RetVal )
            {
                case RETURNCANCEL:   curr = NULL;
                case RETURNKEY:
                case RETURNMOUSE:    goto quit;
                case RETURNPENDING:  break;
            }
        }

        else if( curr->Flags & (CHECKGADGET|OPTIONGADGET) )
        {
            ie = _editcheckgadget( curr, first, left_off, top_off );
        }

        else if( curr->Flags & POPGADGET )
        {
            ie = _editpopgadget( curr, first, left_off, top_off );
        }

        else if( curr->Flags & CUSTOMGADGET )
        {
            (*gadhandler)( first, curr, left_off, top_off, CG_ACTIVATED );
        }

        if( ie->buttons & MBUTTON_LEFT )        /* mouse event */
        {
            /* find the new gadget, if any */
            new = FindGadget( first, ie->mousex - left_off, ie->mousey - top_off );
            nowait = FALSE;
again:
            if( new != NULL )
            {
                /* klicked on new gadget */
                if( curr != new && curr )
                {
                    /* klicked on new gadget, refresh old one, if any */
                    _drawgadget( curr, left_off, top_off, _NOTCURRGAD );
                }
                curr = new;
                if( curr->Flags & (BOOLGADGET|CHECKGADGET|OPTIONGADGET) )
                {
                    /* the new gadget is a bool gadget */
                    if( curr->Flags & SELECTED ) {
                        if( !(curr->Flags & ENDGADGET)
                            && !(curr->Flags & NODESELECT) ) {
                            /* is was selected, deselect it */
                            CLEARBIT(curr->Flags, SELECTED);
                            if( gadhandler )
                                (*gadhandler)( first, curr, left_off, top_off,
                                                CG_CHANGE );
                        }
                    } else {
                        /* it was not selected */
                        if( curr->MutalExclude ) {
                            for( gp = first; gp; gp = gp->Next )
                            {
                                if( gp->MutalExclude & curr->MutalExclude )
                                {
                                    CLEARBIT(gp->Flags,SELECTED);
                                    if( gadhandler )
                                        (*gadhandler)( first, gp, left_off, top_off,
                                                        CG_CHANGE );
                                    _drawgadget( gp, left_off, top_off, _NOTCURRGAD );
                                }
                            }
                        }
                        SETBIT(curr->Flags,SELECTED);
                        if( gadhandler )
                            (*gadhandler)( first, curr, left_off, top_off,
                                            CG_CHANGE );
                    }
                    /* draw the new one as a current gadget */
                    _drawgadget( curr, left_off, top_off, _CURRGADGET );

                    if( !nowait ) {
                        /* wait for user to release mouse button, this looks
                           good! */
                        do
                        {
                            ie = GetInput();
                            new = FindGadget( first, ie->mousex - left_off,
                                               ie->mousey - top_off );
                            if( new != curr ) {
                                if( ! (curr->Flags & NODESELECT ) )
                                    CLEARBIT(curr->Flags,SELECTED);
                                goto again;
                            }
                        } while( ie->buttons & MBUTTON_LEFT );
                    }

                    if( curr->Flags & ENDGADGET )
                    {
                        if( ! ( curr->Flags & TOGGLESELECT ) )
                        {
                            CLEARBIT(curr->Flags,SELECTED);
                        }
                        goto quit;
                    }
                }
            }
            else /* new == NULL */
            {
                /* the new gadget is not found, e.g. the user klicked
                   somewhere else on screen, draw the current gadget
                   as a noncurrent one and clear 'curr', so no gadget
                   is active now. */
                if( curr )
                {
                    _drawgadget( curr, left_off, top_off, _NOTCURRGAD );
                }
                curr = NULL;
            }
            continue;
        }

        if( ie->key )
        {
            /* process keyboard input */
            switch( ie->key )
            {
                case BK_ESC:    curr = NULL;
                                goto quit;
                case BK_RETURN: if( curr->Flags & BOOLGADGET ) {
                                    if( curr->Flags & ENDGADGET ) {
                                        if( curr->MutalExclude ) {
                                            for( gp = first; gp; gp = gp->Next ) {
                                                if( gp->MutalExclude & curr->MutalExclude ) {
                                                    gp->Flags &= ( ~SELECTED );
                                                    _drawgadget( gp, left_off, top_off, _NOTCURRGAD );
                                                }
                                            }
                                        }
                                        curr->Flags |= SELECTED;
                                        _drawgadget( curr, left_off, top_off, _CURRGADGET );
                                        if( ! ( curr->Flags & TOGGLESELECT ) ) {
                                            curr->Flags &= ~SELECTED;
                                        }
                                    }
                                }
                                goto quit;
                case BK_TAB:
                                gp = curr;
                                while( 1 ) {
                                    gp = gp->Next;
                                    if( gp == NULL )
                                        gp = first;
                                    if( gp == curr )
                                        break;
                                    if( !(gp->Flags & DISABLED ) ) {
                                        _drawgadget( curr, left_off, top_off, _NOTCURRGAD );
                                        curr = gp;
                                        break;
                                    }
                                }
                                break;
                case BK_BACKTAB:
                                gp = _parentgadget( first, curr );
                                if( gp == NULL )
                                    gp = _lastgadget( first );
                                _drawgadget( curr, left_off, top_off, _NOTCURRGAD );
                                curr = gp;
                                break;
                case BK_RIGHT:
                case BK_DOWN:   _drawgadget( curr, left_off, top_off, _NOTCURRGAD );
                                if( curr->Next &&
                                    (curr->Next->Flags & (_GADGETTYPES|DISABLED)) == (curr->Flags & _GADGETTYPES)
                                  ) {
                                    do
                                        curr = curr->Next;
                                    while( curr->Next &&
                                        (curr->Next->Flags & _GADGETTYPES)
                                        == (curr->Flags & _GADGETTYPES)
                                        && curr->Flags & DISABLED );
                                } else {
                                    do {
                                        gp = curr;
                                        curr = _parentgadget( first, curr );
                                    } while( (curr->Flags & (_GADGETTYPES|DISABLED))
                                        == (gp->Flags & _GADGETTYPES) );
                                    curr = gp;
                                }
                                break;
                case BK_LEFT:
                case BK_UP:     _drawgadget( curr, left_off, top_off, _NOTCURRGAD );
                                gp = _parentgadget( first, curr );
                                if( (gp->Flags & _GADGETTYPES) == (curr->Flags & _GADGETTYPES) )
                                    curr = gp;
                                else {
                                    while( curr->Next &&
                                        (curr->Next->Flags & (_GADGETTYPES|DISABLED)) ==
                                        (curr->Flags & _GADGETTYPES) )
                                        curr = curr->Next;
                                }
                                break;
                default:
                                /* find the new gadget, if any */
                                if( new = _findkeygadget( first, ie->key ) ) {
                                    nowait = TRUE;
                                    if( !(new->Flags & HOTKEYNOSELECT) )
                                        goto again;
                                    /* selected new gadget, if any */
                                    if( curr != new && curr )
                                    {
                                        /* selected new gadget, refresh old one, if any */
                                        _drawgadget( curr, left_off, top_off, _NOTCURRGAD );
                                    }
                                    curr = new;
                                }
            }
        }
    }   /* for( ;; ) */

quit:

    /* reset cursor position and size */
    SetCursorSize( sav.Start, sav.End );
    SetCursorPos( sav.XPos, sav.YPos );
    gadhandler = oldhandler;

    ie->key = 0;
    ie->buttons = 0;

    return( curr );
}
// #pragma optimize( "", on )

static struct Gadget *_parentgadget( struct Gadget *first, struct Gadget *g ) {

    struct Gadget *parent, *scan;

    parent = NULL;
    scan = _firstgadget( first );
    if( scan == g )
        return NULL;

    for( scan = _firstgadget( first ); scan; ) {
        if( !(scan->Flags & DISABLED) )
            parent = scan;
        if( scan->Next == g )
            break;
        scan = scan->Next;
    }

    return parent;
}

static struct Gadget *_lastgadget( struct Gadget *first ) {

    struct Gadget *last, *scan;

    for( last = scan = _firstgadget( first ); scan; scan = scan->Next ) {
        if( !(scan->Flags & DISABLED) )
            last = scan;
    }

    return last;
}

static struct Gadget *_firstgadget( struct Gadget *first ) {

    struct Gadget *f;

    for( f = first; f; f = f->Next )
        if( !(f->Flags & DISABLED) )
            break;

    return f;
}

#define BETWEEN( a, b, c )  ( (a) <= (b) && (b) <= (c) )


struct Gadget *FindGadget( struct Gadget *f, int x, int y )
{
    register struct Gadget *g;

    for( g = f; g ; g = g->Next )
    {
        if( g->Flags & DISABLED )
            continue;

        if( BETWEEN( g->Left, x, g->Left + g->Width - 1 )
            && BETWEEN( g->Top, y, g->Top + g->Height - 1 ) )
            {
                return g;
            }
    }
    return NULL;
}

struct Gadget *_findkeygadget( struct Gadget *f, unsigned int key )
{
    register struct Gadget *g;
    char *cp;
    char c;

    c = _altname( key );
    if( !c ) c = (char)key;

    for( g = f; g ; g = g->Next )
    {
        if( g->Flags & DISABLED )
            continue;

        if( g->IText ) {
            cp = strrchr( g->IText->Text, (int)MENUHOTINTRODUCER );
            if( cp && tolower(cp[1]) == tolower(c) )
                return g;
        }
    }
    return NULL;
}


void DrawGadget( struct Gadget *g, unsigned left_off, unsigned top_off ) {

    _drawgadget( g, left_off, top_off, _NOTCURRGAD );
}


static void _drawgadget( struct Gadget *g, int left_off, int top_off, unsigned int type )
{
    int i;
    struct IntuiText *it;
    struct Request *r;
    char *cp;
    char defbuf[12];

    static unsigned char checkmarks[] = "[X]()";

    HideMouse();

    if( gadhandler )
        (*gadhandler)( g, g, left_off, top_off, CG_REDRAW );

    if( g->Flags & BOOLGADGET )
    {
        if( g->Flags & SELECTED ) {
            _printchar( '', g->Width, ibase.GadgetSelect, g->Left + left_off, g->Top + top_off );
            for( i = g->Height-2; i > 0; --i )
               _printchar( ' ', g->Width, ibase.GadgetSelect, g->Left + left_off, g->Top + top_off + i );
            _printchar( '', g->Width, ibase.GadgetSelect, g->Left + left_off, g->Top + top_off + g->Height - 1 );

            it = g->IText;
            _drawgadstring( it->Text,
                            g->Left + it->Left + left_off + 1,
                            g->Top + it->Top + top_off +1,
                            ibase.GadgetSelect, ibase.GadgetHotSelect );
        } else {
            _drawborder( g->Left + left_off, g->Top + top_off,
                         g->Width, g->Height,
                         ( g->Flags & 0xf0000000L ) | ( type == _CURRGADGET ? DOUBLE_BORDER : 0L ) ,
                         g->Flags & DISABLED ? CR_GADGETDISABLE : ibase.GadgetColor,
                         0, 0 );
            for( i = g->Height-2; i > 0; --i )
                _printchar( ' ', g->Width-2, ibase.GadgetColor, g->Left + left_off + 1, g->Top + top_off + i );

            it = g->IText;
            _drawgadstring( it->Text,
                            g->Left + it->Left + left_off + 1,
                            g->Top + it->Top + top_off + 1,
                            ibase.GadgetColor, ibase.GadgetHot );
        }
    }
    else if( g->Flags & REQGADGET )
    {
        _drawborder( g->Left + left_off ,g->Top + top_off,
                     g->Width - 1, g->Height,
                     type & _CURRGADGET ? DOUBLE_BORDER : 0L,
                     ibase.WindowColor, 0, 0 );
        _drawrequest( g, left_off, top_off );
    }
    else if( g->Flags & POPGADGET )
    {
        if( g->IText )
        {
            it = g->IText;
            _drawgadstring( it->Text,
                            g->Left + it->Left + left_off,
                            g->Top + it->Top + top_off,
                            ibase.WindowColor, ibase.GadgetHot );
        }

        r = (struct Request _far *)g->SpecialInfo;

        VideoOut( ibase.WindowColor, '', g->Left+left_off, g->Top+top_off );
        VideoOut( CR_LIST, ' ', g->Left+left_off+1, g->Top+top_off );
        _nprint( r->Items[ r->CurrIt ]->Text, g->Width-4,
                    CR_LIST, g->Left + left_off + 2,g->Top + top_off);
        VideoOut( ibase.WindowColor, '', g->Left+left_off+g->Width-2, g->Top+top_off );
        VideoOut( CR_LISTKNOB, _GADARRDOWN,
                  g->Left+g->Width+left_off-1, g->Top+top_off );
    }
    else if( g->Flags & ( TEXTGADGET | LONGINT ))
    {
        if( g->IText )
        {
            it = g->IText;
            _drawgadstring( it->Text,
                            g->Left + it->Left + left_off,
                            g->Top + it->Top + top_off,
                            ibase.WindowColor, ibase.GadgetHot );
            if( it->Next )
                PrintIText( it->Next, g->Left+left_off, g->Top+top_off );
        }
        if( ! ( g->Flags & BORDERLESS ) )
        {
            _drawborder( g->Left + left_off - 1, g->Top + top_off - 1,
                g->Width + 2, g->Height + 2,
                g->Flags & 0xf0000000L, CR_TEXTGADGET,0,0 );
        }
        cp = ((struct StringInfo _far *)g->SpecialInfo)->Buffer;
        if( !cp )
            cp = defbuf;
        if( g->Flags & LONGINT )
            ltoa(((struct StringInfo _far *)g->SpecialInfo)->LongInt, cp, 10 );

        _nprint( cp, g->Width, CR_TEXTGADGET, g->Left + left_off,
            g->Top + top_off);
        if( g->Flags & MARKEND ) {
            /* look if end of string is visible */
            if( strlen(cp) < g->Width ) {
                _printchar( '', 1, ibase.WindowColor,
                    g->Left + left_off + strlen(cp), g->Top + top_off );
            }
        }
    }
    else if( g->Flags & (CHECKGADGET|OPTIONGADGET) )
    {
        if( g->IText )
        {
            it = g->IText;
            _drawgadstring( it->Text,
                            g->Left + it->Left + left_off,
                            g->Top + it->Top + top_off,
                            ibase.WindowColor, ibase.GadgetHot );
        }
        if( g->Flags & CHECKGADGET )
            i = 0;
        else
            i = 3;

        VideoOut( ibase.GadgetColor, checkmarks[i],
                  g->Left + left_off, g->Top + top_off );
        VideoOut( ibase.GadgetColor, checkmarks[i+2],
                  g->Left + left_off + 2, g->Top + top_off );
        if( g->Flags & SELECTED )
            VideoOut( ibase.GadgetColor, checkmarks[i+1],
                  g->Left + left_off + 1, g->Top + top_off );
        else
            VideoOut( ibase.GadgetColor, ' ',
                  g->Left + left_off + 1, g->Top + top_off );
    }
    ShowMouse();
}

static int _drawgadstring( str, x, y, color, hotcolor )
char *str;
int x, y;
colorreg color, hotcolor;
{
    char *cp;
    int len, length;

    /* Search for introducer */
    cp = strrchr( str, (int)MENUHOTINTRODUCER );
    /* get length of string */
    length = strlen( str );

    if( cp ) {
        /* introducer found */
        --length;
        len = cp - str;
        /* Print first part */
        _nprint( str, len, color, x, y );
        x += len;
        /* print hotkey */
        VideoOut( hotcolor,*++cp, x++, y );
        ++cp;
    } else
        cp = str;

    /* print rest */
    VideoStr( color, cp, x, y );

    return length;
}



static struct InputEvent *_editboolgadget( struct Gadget *g , struct Gadget *first, int left_off, int top_off )
{
    struct Gadget *gp;
    struct InputEvent *ie;
    unsigned int lo, to;

    _drawgadget( g, left_off, top_off, _CURRGADGET );

    SetCursorSize( 7, 8 );
    lo = g->Left + left_off + 1;
    to = g->Top + top_off + 1;
    if( g->IText ) {
        lo += g->IText->Left;
        to += g->IText->Top;
    }
    SetCursorPos( lo, to );

    for( ;; )
    {
        ie = ReadInput();
        if( ie->buttons & MBUTTON_LEFT )
        {
            ClearCursor();
            return( ie );
        }
        if( ie->key )
        {
            break;
        }
    }

    switch( ie->key )
    {
        case BK_SPACE:
        case BK_INS:
                        if( !(g->Flags & ENDGADGET) ) {
                            if( g->MutalExclude )
                            {
                                for( gp = first; gp; gp = gp->Next )
                                {
                                    if( gp->MutalExclude & g->MutalExclude )
                                    {
                                        CLEARBIT(gp->Flags,SELECTED);
                                        _drawgadget( gp, left_off, top_off, _NOTCURRGAD );
                                    }
                                }
                            }
                            SETBIT( g->Flags, SELECTED );
                            break;
                        }
        case BK_DEL:    if( g->Flags & NODESELECT )
                        {
                            break;
                        }
                        CLEARBIT( g->Flags, SELECTED );
        default:        break;
    }
    ClearCursor();
    return( ie );
}


static struct InputEvent *_editcheckgadget( g , first, left_off, top_off )
struct Gadget *g , *first;
int left_off, top_off;
{
    struct Gadget *gp;
    struct InputEvent *ie;

    _drawgadget( g, left_off, top_off, _CURRGADGET );

    SetCursorSize( 7, 8 );
    SetCursorPos( g->Left + left_off + 1, g->Top + top_off );

    for( ;; )
    {
        ie = ReadInput();
        if( ie->buttons & MBUTTON_LEFT )
        {
            ClearCursor();
            return( ie );
        }
        if( ie->key )
        {
            break;
        }
    }

    switch( ie->key )
    {
        case BK_SPACE:
                        if( g->Flags & TOGGLESELECT
                         && g->Flags & SELECTED
                         && !(g->Flags & NODESELECT) ) {
                            if( gadhandler )
                                (*gadhandler)( first, g, left_off, top_off,
                                                CG_CHANGE );
                            CLEARBIT( g->Flags, SELECTED);
                            break;
                        }
                        // fall thru
        case BK_INS:
                        if( !(g->Flags & ENDGADGET) ) {
                            if( g->MutalExclude )
                            {
                                for( gp = first; gp; gp = gp->Next )
                                {
                                    if( gp->MutalExclude & g->MutalExclude )
                                    {
                                        CLEARBIT(gp->Flags,SELECTED);
                                        if( gadhandler )
                                            (*gadhandler)( first, gp, left_off, top_off,
                                                            CG_CHANGE );
                                        _drawgadget( gp, left_off, top_off, _NOTCURRGAD );
                                    }
                                }
                            }
                            SETBIT( g->Flags,SELECTED);
                            if( gadhandler )
                                (*gadhandler)( first, g, left_off, top_off,
                                                CG_CHANGE );
                            break;
                        }
        case BK_DEL:    if( g->Flags & NODESELECT )
                        {
                            break;
                        }
                        CLEARBIT(g->Flags,SELECTED);
                        if( gadhandler )
                            (*gadhandler)( first, g, left_off, top_off, CG_CHANGE );
        default:        break;
    }
    ClearCursor();
    return( ie );
}


#define adv_isprint( c )  ( (unsigned char)(c) >= 0x1c && (unsigned char)(c) <= 0xfe )

// #pragma optimize( "egl", off )
static struct InputEvent *_edittextgadget( struct Gadget *g, struct Gadget *f, int left_off, int top_off)
{
    int attr = CR_TEXTGADGET;
    int x = 0, i;
    unsigned int firstchar = 1;
    unsigned char *cp, *ccp, *end, *base, *buffer;
    struct InputEvent *ie, *waitmouse;
    struct Gadget *gp;
    struct StringInfo _far *si;
    struct _request _far *_r;

    static char edbuf[12];

    static void _drawtextgadstr(struct Gadget *, char *, char *, int, int,
                                int, colorreg );

    si = (struct StringInfo _far *)(g->SpecialInfo);
    if( si->Buffer )
        buffer = si->Buffer;
    else {
        buffer = edbuf;
        si->MaxChars = _MIN( si->MaxChars, 11 );
    }

    base = cp = buffer;

    if( g->Flags & LONGINT )
        ltoa( si->LongInt, buffer, 10 );

    for( end = cp; *end != '\0'; ++end );
    SetCursorSize( 7, 8 );

    SetCursorPos( g->Left + left_off + x, g->Top + top_off );
    do
    {
        waitmouse = GetInput();
    } while( waitmouse->buttons != 0 );

    _drawtextgadstr(g,buffer,base,left_off,top_off,x,CR_TEXTGADACTIVATED );

    for ever {

        SetCursorPos( g->Left + left_off + x, g->Top + top_off );

        ie = ReadInput();

        if( ie->buttons & MBUTTON_LEFT ) {
            gp = FindGadget( f, ie->mousex - left_off, ie->mousey - top_off );
            if( gp != g )
                goto done;

            do {
                waitmouse = GetInput();
                x = waitmouse->mousex - left_off - g->Left;
                if( x < 0 )
                    x = 0;

                if( x >= g->Width )
                    x = g->Width-1;

                if( x > end - base )
                    x = end - base;

                SetCursorPos( g->Left + left_off + x, g->Top + top_off );
            } while( waitmouse->buttons );
            cp = base + x;
        }
        if( ie->key )
        {
            switch( ie->key )
            {
                case 0x0000:    break;
                case BK_UP:
                case BK_DOWN:
                case BK_TAB:
                case BK_BACKTAB:
                case BK_RETURN:
                case BK_ESC:    goto done;

                case BK_LEFT:   if( x ) {
                                    --x;
                                    --cp;
                                }
                                else if( base > buffer ) {
                                    --base;
                                    --cp;
                                }
                                break;
                case BK_RIGHT:  if( *cp ) {
                                    if( x + 1 == g->Width )
                                        ++base;
                                    else
                                        ++x;
                                    ++cp;
                                }
                                break;
                case BK_HOME:   x = 0;
                                cp = base = buffer;
                                break;
                case BK_END:    x = end - base;
                                cp = end;
                                if( x > g->Width ) {
                                    x = g->Width - 1;
                                    base = end - x;
                                }
                                break;
                case BK_BACKSP: if( cp == buffer )
                                    break;
                                --x;
                                --cp;
                                if( x < 0 ) {
                                    x = 0;
                                    --base;
                                }
                case BK_DEL:    if( *cp ) {
                                    for( ccp = cp; ccp < end; ++ccp )
                                        *ccp = *( ccp + 1 );
                                    --end;
                                }
                                break;
                case BK_F4:
                                if( gadhandler )
                                    (*gadhandler)( f, g, left_off, top_off, CG_F4 );

                                x = 0;
                                cp = base = buffer;
                                end = cp + strlen(cp);
                                break;
                default:        if( ! adv_isprint( ie->key ) ) {
                                    if( _findkeygadget( f, ie->key ) )
                                        goto done;

                                    Sound( ibase.soundfreq, ibase.sounddur );
                                    break;
                                }
                                if( g->Flags & LONGINT ) {
                                    if( (char)ie->key == '+' ) {
                                        if( buffer[0] == '-' ) {
                                            for( ccp = buffer; ccp < end; ++ccp )
                                                *ccp = *(ccp + 1 );
                                            --end;
                                            if( cp > buffer ) --cp;
                                            if( x ) --x;
                                        }
                                        break;
                                    } else if( (char)ie->key == '-' ) {
                                        if( buffer[0] != '-' ) {
                                            for( ccp = end + 1; ccp > buffer; --ccp )
                                                *ccp = *( ccp - 1 );
                                            *buffer = '-';
                                            ++end;
                                            ++cp;
                                            ++x;
                                            if( x >= g->Width ) {
                                                --x;
                                                ++base;
                                            }
                                        }
                                        break;
                                    } else if( !isdigit( (char)ie->key ) ) {
                                        Sound( ibase.soundfreq, ibase.sounddur );
                                        break;
                                    }
                                }
                                if( firstchar ) {
                                    buffer[0] = '\0';
                                    end = buffer;
                                    HideMouse();
                                    _printchar( ' ', g->Width, attr, g->Left + left_off, g->Top + top_off );
                                    ShowMouse();
                                }
                                if( ( end - buffer ) < si->MaxChars ) {
                                    for( ccp = end + 1; ccp > cp; --ccp )
                                        *ccp = *( ccp - 1 );
                                    *ccp = (char)ie->key;
                                    ++end;
                                    ++cp;
                                    ++x;
                                    if( x >= g->Width ) {
                                        --x;
                                        ++base;
                                    }
                                }
            }
        }

        if( firstchar ) {
            firstchar = 0;
        }

        _drawtextgadstr(g,buffer,base,left_off,top_off,x,CR_TEXTGADGET);
    }
done:
    if( g->Flags & LONGINT ) {
        si->LongInt = atol( buffer );
    }
    ClearCursor();
    return( ie );
}
// #pragma optimize( "", on )

static void _drawtextgadstr(g,buffer,base,left_off,top_off,x,color )
struct Gadget *g;
char *buffer, *base;
int left_off, top_off, x;
colorreg color;
{

    int i;

    HideMouse();
    /* nprint doesn't work because we have different attributes for
       text background and blanked background */
    _printl( base, g->Width, color,
        g->Left + left_off, g->Top + top_off );

    /* look if end of string is visible */
    i = strlen(base);
    if( i < g->Width ) {
        if( g->Flags & MARKEND ) {
            _printchar( '', 1, ibase.WindowColor,
                g->Left + left_off + i, g->Top + top_off );
            ++i;
        }
        _printchar( ' ', g->Width-i, CR_TEXTGADGET, g->Left + left_off + i, g->Top + top_off );
    }
    ShowMouse();
}

static struct InputEvent *_editpopgadget( struct Gadget *g, struct Gadget *f, int left_off, int top_off)
{
    struct InputEvent *ie;

    struct Gadget text, button, list, *curr;
    struct Request *r;
    struct Window w;
    BOOL winopen = FALSE;
    unsigned int numit, currit;

    text = *g;
    --text.Width;
    text.Next = &button;

    /* make expand button */
    button = *g;
    button.Left += g->Width-1;
    button.Width = 1;
    button.Next = NULL;

    ie = ibase.ie;
    r = (struct Request *)g->SpecialInfo;
    for( numit = 0; r->Items[numit]; ++numit );

    list.Next = NULL;
    list.IText = NULL;
    list.Flags = SINGLESELECT|ENDGADGET|REQGADGET;
    list.SpecialInfo = (APTR)r;
    list.MutalExclude = 0L;

    /* setup window */
    list.Left = w.NewLeft = g->Left + left_off;
    list.Top = w.NewTop =  g->Top + top_off + 1;
    list.Width = w.Width = g->Width;
    list.Height = w.Height = 6;
    w.Title = NULL;
    w.Flags = SHADOWLESS|BORDERLESS;

    if( ie->buttons )
        curr = FindGadget( &text, ie->mousex - left_off, ie->mousey - top_off );
    else
        curr = &text;

    do {
        /* loop once per action */

        if( curr == &text ) {
            /* clicked on text line */
            SetCursorSize( 7, 8 );
            SetCursorPos( g->Left + left_off, g->Top + top_off );

            ie = ReadInput();
            if( ie->key ) {
                switch( ie->key ) {
                    case BK_UP: if( r->CurrIt ) {
                                    --r->CurrIt;
                                    if( gadhandler )
                                        (*gadhandler)( f, g, left_off, top_off, CG_CHANGE );
                                }
                                break;
                    case BK_DOWN: if( r->CurrIt+1 < numit ) {
                                    ++r->CurrIt;
                                    if( gadhandler )
                                        (*gadhandler)( f, g, left_off, top_off, CG_CHANGE );
                                }
                                break;
                    case BK_F4:   curr = &button;
                                  continue;
                    case BK_RETURN:
                    case BK_ESC:
                    case BK_TAB:
                    case BK_BACKTAB:
                                  goto done;
                    default:
                        if( _findkeygadget( f, ie->key ) != g ) {
                            ClearCursor();
                            goto done;
                        }
                }
            }

            ClearCursor();

        } else if( curr == &button ) {
            /* clicked on expand button */
            if( winopen ) {
                /* already open, close window */
                CloseWindow( &w );
                winopen = FALSE;
                while( ie->buttons & MBUTTON_LEFT ) ie = GetInput();
            } else if( OpenWindow( &w, 0 ) ) {
                winopen = TRUE;
                _drawgadget( &list, 0, 0, _CURRGADGET );
                while( ie->buttons & MBUTTON_LEFT ) ie = GetInput();

                ie->key = BK_TAB;
                currit = r->CurrIt;

                _openrequest( &list, &list, 0, 0 );

                switch( r->RetVal ) {
                    case RETURNMOUSE:
                        while( ie->buttons & MBUTTON_LEFT )
                            ie = GetInput();
                    case RETURNKEY:
                        CloseWindow( &w );
                        winopen = FALSE;
                        if( gadhandler )
                            (*gadhandler)( f, g, left_off, top_off, CG_CHANGE );
                        break;
                    case RETURNCANCEL:
                        goto done;
                    case RETURNPENDING:
                        r->CurrIt = currit;
                        switch( ie->key ) {
                            case BK_TAB:
                            case BK_BACKTAB:
                                          goto done;
                        }
                        break;
                }
            }
        }

        if( ie->buttons & MBUTTON_LEFT ) {
            curr = FindGadget( g, ie->mousex - left_off, ie->mousey - top_off );
            if( curr != g ) goto done;
            curr = FindGadget( &text, ie->mousex - left_off, ie->mousey - top_off );
        } else
            curr = &text;

        _drawgadget( g, left_off, top_off, _CURRGADGET );
    }while( curr );
done:
    if( winopen ) CloseWindow( &w );

    return ie;
}


static int _findboxtop( int, int, int, int );
static int _findknobpos( int, int, int, int );
static int _drawknob( struct Gadget *, int , int , int , int , int );
static void _drawbox( struct Gadget *, int, int, int, int );
static void _drawrequest( struct Gadget *, int, int );

// #pragma optimize( "elg", off )
static struct InputEvent *_openrequest( struct Gadget *f, struct Gadget *req, int left_off, int top_off )
{
    struct Gadget g_up, g_down, g_box, g_prop, g_req;
    struct _request _far *r;
    BOOL change = FALSE;

    int curr, new, boxtop, numit, knobheight, knobpos, firsttime = 1, i;
    struct InputEvent *ie;
    struct IntuiText *it;

    ie = ibase.ie;  /* initialize pointer */
    g_req = *req;   /* copy structure */
    g_req.Next = NULL;

    r = (struct _request _far *)(req->SpecialInfo);

    g_up.Next = &g_req;
    g_up.IText = g_down.IText = g_box.IText = g_prop.IText = NULL;
    g_up.Width = g_down.Width = g_prop.Width = 2;
    g_up.Height = g_down.Height = 1;
    g_up.Flags = g_down.Flags = g_box.Flags = g_prop.Flags =
                                BOOLGADGET | ENDGADGET | BORDERLESS;
    g_down.Next = &g_up;

    g_prop.Next = &g_box;
    g_prop.Left = g_up.Left = g_down.Left = req->Left + req->Width - 1;
    g_up.Top = req->Top;
    g_down.Top = req->Top + req->Height - 1;
    g_prop.Top = req->Top + 1;
    g_prop.Height = req->Height - 2;

    g_box.Next = &g_down;
    g_box.Left = req->Left + 1;
    g_box.Top = req->Top + 1;
    g_box.Width = req->Width - 3;
    g_box.Height = req->Height - 2;

    if( r->Flags & OWNREDRAW )
        numit = r->NumIt;
    else
        for( numit = 0; r->Items[numit]; ++numit );

    curr = r->CurrIt;
    boxtop = r->BoxTop;
    while( curr >= g_box.Height + boxtop )
        ++boxtop;
    r->BoxTop = boxtop;

    new = -1;

    if( numit ) {
        knobheight = ( g_box.Height * g_box.Height ) / numit;
        if( knobheight == 0 ) {
            knobheight = 1;
        }
        else if( knobheight > g_box.Height ) {
            knobheight = g_box.Height;
        }
    } else
        knobheight = g_box.Height;

    _drawgadget( req, left_off, top_off, _CURRGADGET );

    if( numit == 0 ) {
        /* no items */
        ie = ReadInput();
        r->RetVal = RETURNPENDING;
        switch( ie->key ) {
            case BK_RETURN: r->RetVal = RETURNKEY;
                            break;
            case BK_ESC:    r->RetVal = RETURNCANCEL;
        }
        return ie;
    }

    knobpos = _findknobpos( g_box.Height, boxtop, numit, knobheight );
    for( ;; )
    {
        if( new != -1 )
        {
            curr = new;
            /* 'curr' keeps track of the current Item */
            r->CurrIt = curr;
            if( change ) {
                change = FALSE;
                if( gadhandler )
                    (*gadhandler)( f, req, left_off, top_off, CG_CHANGE );
            }
            r->BoxTop = boxtop;
            HideMouse();
            _drawbox( req, boxtop, curr, left_off, top_off );
            knobpos = _drawknob( req, knobheight, boxtop, numit, left_off, top_off );
            ShowMouse();
        }
        if( firsttime == 0 || ie->key )
        {
            ie = ReadInput();
        }
        firsttime = 0;

        if( r->Flags & OWNREDRAW && gadhandler ) {
            (*gadhandler)( f, req, left_off, top_off, CG_OWNREDRAW|CG_CHANGE );
            new = curr = r->CurrIt;
            boxtop = r->BoxTop;
        } else
            new = -1;

        if( ie->key )
        {
            switch( ie->key )
            {
                case BK_SPACE:
                            if( r->Flags & MULTISELECT ) {
                                new = curr;   /* mark to refresh */
                                it = r->Items[curr];
                                if( it->Flags & IT_SELECTED )
                                    CLEARBIT(it->Flags, IT_SELECTED);
                                else {
                                    SETBIT(it->Flags, IT_SELECTED);
                                    SETBIT( r->Flags, ISMULTISEL );
                                }
                            }
                            /* fall thru */
                case BK_DOWN:    /* down arrow */
                            if( curr+1 < numit )
                            {
                                change = TRUE;
                                new = curr + 1;
                                if( new >= g_box.Height + boxtop )
                                    ++boxtop;
                            }
                            break;
                case BK_UP:    /* up arrow */
                            if( curr )
                            {
                                change = TRUE;
                                new = curr - 1;
                                if( new < boxtop )
                                    --boxtop;
                            }
                            break;
                case BK_PGDOWN:   /* page down */
                            change = TRUE;
                            boxtop = _MIN( numit - (int)g_box.Height,
                                        boxtop + g_box.Height - 1 );
                            if( boxtop < 0 ) boxtop = 0;

                            new = _MIN( numit-1, boxtop + g_box.Height - 1 );
                            if( new < 0 ) new = 0;

                            break;
                case BK_PGUP:   /* page up */
                            change = TRUE;
                            boxtop = _MAX( 0, boxtop - (int)g_box.Height + 1 );
                            new = boxtop;
                            break;
                case BK_LEFT:
                            /* BK_LEFT wouldn't leave group */
                            ie->key = BK_BACKTAB;
                            /* fall thru */
                case BK_BACKTAB:
                            r->RetVal = RETURNPENDING;
                            goto done;
                case BK_RIGHT:
                            ie->key = BK_TAB;
                            /* fall thru */
                case BK_TAB:
                            r->RetVal = RETURNPENDING;
                            goto done;
                case BK_ESC:
                            r->RetVal = RETURNCANCEL;
                            goto done;
                case BK_RETURN:
                            change = TRUE;
                            r->RetVal = RETURNKEY;
                            goto done;
                default:
                            if( adv_isprint( ie->key ) ) {
                                for( i = curr+1;; ++i ) {
                                    if( tolower( (int)r->Items[i]->Text[0] )
                                        == tolower( (char)ie->key ) )
                                        break;

                                    if( i >= numit ) {
                                        for( i = curr+1; i <= numit; ++i ) {
                                            if( tolower( (int)r->Items[i]->Text[0] )
                                                >= tolower( (char)ie->key ) )
                                                break;
                                        }
                                        break;
                                    }
                                }

                                if( i < numit ) {
                                    change = TRUE;
                                    new = i;
                                    boxtop = _MAX( boxtop, new - g_box.Height + 1 );
                                }
                            } else if( _altname( ie->key ) ) {
                                if( _findkeygadget( f, ie->key ) )
                                    goto done;
                            }
                            break;
            }
        }
        if( ie->buttons & MBUTTON_LEFT )
        {
            struct Gadget *g;
            unsigned long time, passed;

            g = FindGadget( &g_prop, ie->mousex - left_off, ie->mousey - top_off );
            if( g == &g_box )
            {
                new = boxtop + ie->mousey - top_off - g_box.Top;
                if( new >= numit ) {
                    new = -1;
                } else {
                    if( ( new == curr
                        && ie->buttons & DOUBLE_CLICK( MBUTTON_LEFT ) )
                        || req->Flags & SINGLESELECT ) {
                            r->RetVal = RETURNMOUSE;
                            if( r->CurrIt != new ) {
                                r->CurrIt = new;
                                if( gadhandler )
                                    (*gadhandler)( f, req, left_off, top_off, CG_CHANGE );
                            }
                            r->BoxTop = boxtop;
                            ClearCursor();
                            return( ie );

                        new = -1;
                    } else if( r->Flags & MULTISELECT ) {
                        it = r->Items[new];
                        if( it->Flags & IT_SELECTED )
                            CLEARBIT(it->Flags, IT_SELECTED);
                        else {
                            SETBIT(it->Flags, IT_SELECTED);
                            SETBIT( r->Flags, ISMULTISEL );
                        }
                    }
                    /* now any item has been selected, even if the current
                       item has not changed, call gadget handler. */
                    r->CurrIt = new;
                    if( gadhandler )
                        (*gadhandler)( f, req, left_off, top_off, CG_CHANGE );

                }
            }
            else if( g == &g_up )
            {
                if( boxtop )
                {
                    if( curr >= boxtop + g_box.Height - 1 )
                        --curr;
                    --boxtop;
                }
                HideMouse();
                _drawbox( req, boxtop, curr, left_off, top_off );
                knobpos = _drawknob( req, knobheight, boxtop, numit, left_off, top_off );
                ShowMouse();
                time = GetClock();
                do
                {
                    ie = GetInput();
                    passed = GetClock() - time;
                    if( passed > 7 )
                    {
                        time = GetClock();
                        do
                        {
                            ie = GetInput();
                            passed = GetClock() - time;
                            if( passed > 1 )
                            {
                                time = GetClock();
                                if( boxtop )
                                {
                                    if( curr >= boxtop + g_box.Height - 1 )
                                        --curr;
                                    --boxtop;
                                }
                                HideMouse();
                                _drawbox( req, boxtop, curr, left_off, top_off );
                                knobpos = _drawknob( req, knobheight, boxtop, numit, left_off, top_off );
                                ShowMouse();
                            }
                        } while( ie->buttons & MBUTTON_LEFT );
                    }
                } while( ie->buttons & MBUTTON_LEFT );
                new = curr;
                continue;
            }
            else if( g == &g_down )
            {
                if( boxtop < numit - g_box.Height )
                {
                    if( curr == boxtop )
                        curr = curr + 1;
                    ++boxtop;
                HideMouse();
                _drawbox( req, boxtop, curr, left_off, top_off );
                knobpos = _drawknob( req, knobheight, boxtop, numit, left_off, top_off );
                ShowMouse();
                }
                time = GetClock();
                do
                {
                    ie = GetInput();
                    passed = GetClock() - time;
                    if( passed > 7 )
                    {
                        time = GetClock();
                        do
                        {
                            ie = GetInput();
                            passed = GetClock() - time;
                            if( passed > 1 )
                            {
                                time = GetClock();
                                if( boxtop < numit - g_box.Height )
                                {
                                    if( curr == boxtop )
                                        curr = curr + 1;
                                    ++boxtop;
                                    HideMouse();
                                    _drawbox( req, boxtop, curr, left_off, top_off );
                                    knobpos = _drawknob( req, knobheight, boxtop, numit, left_off, top_off );
                                    ShowMouse();
                                }
                            }
                        } while( ie->buttons & MBUTTON_LEFT );
                    }
                } while( ie->buttons & MBUTTON_LEFT );
                new = curr;
                continue;
            }
            else if( g == &g_prop )
            {
                int oldy, i, offset;
                offset = ie->mousey - top_off - g_prop.Top - knobpos;

                change = TRUE;

                if( offset >= 0 && offset < knobheight )
                {
                    HideMouse();
                    for( i = 0; i < knobheight; ++i ) {
                        VideoStr( CR_LISTKNOBSELECT, "",
                            g_up.Left + left_off, g_up.Top + 1 + top_off + i + knobpos );
                    }
                    ShowMouse();
again:
                    for( oldy = ie->mousey; ie->buttons & MBUTTON_LEFT; )
                    {
                        ie = GetInput();

                        if( ie->mousey == oldy ) goto again;
                        knobpos = ie->mousey - top_off - g_prop.Top - offset;
                        if( knobpos < 0 )
                            knobpos = 0;
                        if( knobpos + knobheight > g_prop.Height )
                            knobpos = g_prop.Height - knobheight;

                        oldy = ie->mousey;
                        boxtop = _findboxtop( g_prop.Height, knobpos, numit, knobheight );

                        if( curr < boxtop )
                            curr = boxtop;
                        else if( curr >= boxtop + g_box.Height )
                            curr = boxtop + g_prop.Height - 1;

                        HideMouse();
                        _drawbox( req, boxtop, curr, left_off, top_off );

                        for( i = 0; i < g_prop.Height; ++i )
                        {
                            VideoStr( CR_LISTKNOB, "",
                                g_up.Left + left_off, g_up.Top + 1 + top_off + i );
                        }
                        for( i = 0; i < knobheight; ++i )
                        {
                            VideoStr( CR_LISTKNOBSELECT, "",
                                g_up.Left + left_off, g_up.Top + 1 + top_off + i + knobpos );
                        }
                        ShowMouse();
                    }
                }
                else
                {
                    knobpos = ie->mousey - top_off - (int)g_prop.Top -
                        knobheight/2;
                    knobpos = _MIN( knobpos, (int)g_box.Height - knobheight );
                    knobpos = _MAX( knobpos, 0 );

                    boxtop = _findboxtop( g_box.Height, knobpos, numit, knobheight );

                    if( curr < boxtop )
                        curr = boxtop;
                    else if( curr >= boxtop + g_box.Height )
                        curr = boxtop + g_box.Height - 1;

                    HideMouse();
                    _drawbox( req, boxtop, curr, left_off, top_off );
                    ShowMouse();
                }


                HideMouse();
                for( i = 0; i < g_prop.Height; ++i ) {
                    VideoStr( CR_LISTKNOB, "",
                        g_up.Left + left_off, g_up.Top + 1 + top_off + i );
                }
                for( i = 0; i < knobheight; ++i ) {
                    VideoStr( CR_LISTKNOB, "",
                        g_up.Left + left_off, g_up.Top + 1 + top_off + i + knobpos );
                }
                ShowMouse();

            } /* pseudo switch( gadget ) */
            else if( g == NULL )
            {
                r->RetVal = RETURNPENDING;
                r->CurrIt = curr;
                if( change ) {
                    if( gadhandler )
                        (*gadhandler)( f, req, left_off, top_off, CG_CHANGE );
                }
                r->BoxTop = boxtop;
                ClearCursor();
                return( ie );
            }
        } /* if( ie->buttons ) */
    } /* for( ;; ) */

done:
    r->CurrIt = curr;
    if( change ) {
        if( gadhandler )
            (*gadhandler)( f, req, left_off, top_off, CG_CHANGE );
    }
    r->BoxTop = boxtop;
    ClearCursor();
    return( ie );
}
// #pragma optimize( "", on )

/*
 *    Name: SetMSGadget()
 *  Return: TRUE if Gadget is MULISELECTABLE, FALSE if not.
 * Purpose: This routine set the IT_SELECTED flag in the current
 *          Item of the structure.
 *
 *
 * (c)M.Schollmeyer
 */
BOOL SetMSGadget( struct Request *r ) {

    struct IntuiText **ita;

    ita = r->Items;
    while( *ita ) {
        if( (*ita)->Flags & IT_SELECTED )
            return( r->Flags & MULTISELECT );
        ++ita;
    }

    SETBIT( r->Items[r->CurrIt]->Flags, IT_SELECTED );

    return FALSE;
}


static void _drawrequest( struct Gadget *g, int left_off, int top_off )
{
    int numit, knobheight;
    struct _request _far *r;

    r = (struct _request _far *)(g->SpecialInfo);

    for( numit = 0; r->Items[numit]; ++numit );

    if( r->CurrIt >= numit )
        r->CurrIt = numit-1;

    if( r->CurrIt < r->BoxTop )
        r->BoxTop = r->CurrIt;
    else if( r->CurrIt > r->BoxTop + g->Height - 2 )
        r->BoxTop = _MIN( r->CurrIt, numit - g->Height + 2 );
    else if( numit < r->BoxTop + g->Height - 2 )
        r->BoxTop = numit - g->Height + 2;


    if( numit ) {
        knobheight = (g->Height-2) * (g->Height-2) / numit;
        if( knobheight == 0 )
            knobheight = 1;
        else if( knobheight > g->Height - 2 ) {
            knobheight = g->Height - 2;
            r->BoxTop = 0;
        }
    } else {
        knobheight = g->Height - 2;
        r->BoxTop = 0;
    }

    _drawbox( g, r->BoxTop, r->CurrIt, left_off, top_off );

    VideoOut( CR_LISTKNOB, _GADARRUP, g->Left + g->Width - 1 + left_off, g->Top + top_off);
    VideoOut( CR_LISTKNOB, _GADARRDOWN, g->Left + g->Width - 1 + left_off, g->Top + g->Height - 1 + top_off);

    _drawknob( g, knobheight, r->BoxTop, numit, left_off, top_off );
}


static int _findboxtop( int boxheight, int knobpos, int numit, int knobheight )
{
    int boxtop, search;

    if( knobpos == boxheight - 1 )
    {
        return _MAX( 0, numit - boxheight );
    }

    for( boxtop = 0; boxtop <= numit; ++boxtop )
    {
        search = _findknobpos( boxheight, boxtop, numit, knobheight );
        if( search >= knobpos )
        {
            return( boxtop );
        }
    }
}

static int _findknobpos( int boxheight, int boxtop, int numit, int knobheight )
{
    if( boxtop == 0 )
    {
        return( 0 );
    }
    else if( boxtop == numit - boxheight )
    {
        return( boxheight - knobheight );
    }
    else
    {
        return (int)( (long)boxheight * (long)boxtop / (long)numit );
    }
}

static int _drawknob( struct Gadget *g, int knobheight, int boxtop, int numit, int left_off, int top_off )
{
    int i, knobpos;
    struct _request _far *r;

    r = (struct _request _far *)(g->SpecialInfo);

    knobpos = _findknobpos( g->Height - 2, boxtop, numit, knobheight );

    for( i = 0; i < g->Height-2; ++i )
    {
        VideoStr( CR_LISTKNOB, "",
                g->Left + g->Width - 1 + left_off, g->Top + top_off + 1 + i );
    }
    for( i = 0; i < knobheight; ++i )
    {
        VideoStr( CR_LISTKNOB, "",
                g->Left + g->Width - 1 + left_off, g->Top + top_off + 1 + i + knobpos );
    }
    return( knobpos );
}


static void _drawbox( struct Gadget *g, int top, int curr, int left_off, int top_off )
{
    int i, attr, end = 0;
    struct Request _far *r;
    struct Gadget *lg;
    struct StringInfo _far *lsi;
    struct IntuiText *it;

    HideMouse();

    r = (struct Request _far *)(g->SpecialInfo);

    if( r->Flags & NOBARHILITE ) {
        SetCursorSize( 7, 8 );
    }

    if( r->Flags & OWNREDRAW && gadhandler ) {
        r->CurrIt = curr;
        r->WinTop = top;
        (*gadhandler)( g, g, left_off, top_off, CG_OWNREDRAW );

        ShowMouse();
        return;
    }

    if( *(r->Items) ) {
        for( i = 0; i < g->Height-2; ++i ) {
            it = r->Items[top+i];

            if( !it )
                break;

            if( it->Flags & IT_HILITE )
                attr = CR_LISTHILITE;
            else if( it->Flags & IT_ITALIC )
                attr = CR_LISTITALIC;
            else
                attr = CR_LIST;

            if( i == curr - top ) {
                if( r->Flags & NOBARHILITE )
                    SetCursorPos( g->Left + left_off + 1, g->Top + top_off + i + 1 );
                else
                    attr = CR_LISTBAR;
            }

            if( it->Flags & IT_SELECTED )
                VideoOut( attr, '', g->Left + left_off + 1, g->Top + top_off + i + 1 );
            else
                VideoOut( attr, ' ', g->Left + left_off + 1, g->Top + top_off + i + 1 );


            _nprint( it->Text, g->Width-4, attr,
                    g->Left + left_off + 2, g->Top + top_off + i + 1 );
        }

        for( ; i < g->Height-2; ++i )
            _printchar( ' ', g->Width - 3, CR_LIST,
                g->Left + left_off + 1, g->Top + top_off + i + 1 );

    } else {

        /* No Items */

        _printchar( ' ', g->Width - 3, CR_LIST, g->Left + left_off + 1, g->Top + top_off + 1 );
        _nprint( " No Entries found.", g->Width - 3, CR_LIST, g->Left + left_off + 1, g->Top + top_off + 2 );
        for( i = 3; i < g->Height-1; ++i )
            _printchar( ' ', g->Width - 3, CR_LIST, g->Left + left_off + 1, g->Top + top_off + i );
    }

    ShowMouse();

}


/*
 *      NAME:           CheckMenu
 *      ARGUMENTS:      
 *      RETURN TYPE:    
 *
 *      Purpose:        
 *
 * (c) M. Schollmeyer 1990,1991
 */

static void ( *menuhandler )();


    // flags for _drawmenustring()
#define _DMS_NORMAL 0
#define _DMS_DISABLE 1
#define _DMS_HILITE 2


static int _drawmenustring( char *, unsigned, int, int, int, int );
static unsigned long _menucode( struct Menu *, struct Menu *, struct MenuItem * );
static BOOL _drawmenu( struct Menu *, int );
static void _destroymenu( struct Menu * );
static struct Menu *_findmenu( struct Menu *, int );
static struct MenuItem *_findshortkey( struct Menu *, unsigned int );
static struct MenuItem *_findhotkey( struct Menu *, unsigned int );
static struct InputEvent *_getmenu( struct Menu *, struct Menu *, int, int );

static struct MenuItem *curr;

void DrawMenuTitle( struct Menu *m, unsigned int len, unsigned long flags )
{
    struct Menu *cm;
    int x = 0;
    int attr;

    HideMouse();

    if( flags & MENU_DISABLE )
        attr =  CR_MENUDISABLE;
    else
        attr =  CR_MENU;

    _printchar( ' ', 2, attr, 0, 0 );

    for( cm = m; cm; cm = cm->Next )
    {
                // setup left edge of menu
        cm->buffer[0] = x + 1;
        x += 2;
        if( flags & MENU_DISABLE )
            x += _drawmenustring( cm->MenuName, 0, x, 0, 0, _DMS_DISABLE );
        else {
            x += _drawmenustring( cm->MenuName, 0, x, 0, 0, _DMS_NORMAL );
        }
                // setup right edge of menu
        cm->buffer[1] = x;
        _printchar( ' ', 2, attr, x, 0 );
    }

    if( len > x )
        _printchar( ' ', len-x, attr, x, 0 );

    ShowMouse();

}


unsigned long CheckMenu( struct Menu *m, struct InputEvent *ie,
    void (*handler)() )
{
    struct Menu *cm = NULL, *_m;
    struct MenuItem *mi;
    struct Cursor sav;
    int cancel = FALSE;
    unsigned long accelerator = 0L;

    menuhandler = handler;

    curr = m->First;

    /* first check if we can do something with the input */

    /* check for mouse input */
    if( ie->buttons & MBUTTON_LEFT )
    {
        /* check y-coordinate */
        if( ie->mousey > 0 )
            return MENUNULL;
        /* check x-coordinate */
        if( ie->mousex < m->buffer[0] )
            return MENUNULL;

        /* get last menu */
        for( _m = m; _m->Next; _m = _m->Next );
        if( ie->mousex > _m->buffer[1] )
            return MENUNULL;

    /* check for keyboard input */
    } else if( ie->key ) {

        for( _m = m; _m; _m = _m->Next )
        {
            if( mi = _findshortkey( _m, ie->key ) )
            {
                return (_menucode( m, _m, mi )|ACCELERATORBIT);
            }
        }

        cm = _findmenu( m, ie->key );
        if( ! cm )
            return MENUNULL;

        if( _drawmenu( cm, 0 ) == FALSE )
            return MENUNULL;

    } else {

        return MENUNULL;

    }

    /* now we think we can do something with the input */
    GetCursor( -1, &sav );
    ClearCursor();

    for( ;; )
    {
        if( ie->buttons & MBUTTON_LEFT )
        {
            int keepon = 1; /* keepon is set to 0 if a valid menu has been
                               selected and keeps beeing 1 if the menu
                               hasn't changed */

            if( ie->mousey > 1 )
                goto done;

            while( keepon ) {

                if( ! ie->buttons & MBUTTON_LEFT )
                    break;

                if( ie->mousey == 0 ) {

                    for( _m = m; _m; _m = _m->Next ) {

                        if( ie->mousex >= _m->buffer[0] && ie->mousex <= _m->buffer[1] )
                        {
                            if( _m != cm ) {
                                // new menu selected
                                if( cm )
                                    _destroymenu( cm );
                            }

                            HideMouse();
                            _drawmenustring( _m->MenuName, 0,
                                             _m->buffer[0]+1, 0,
                                             _m->buffer[1] - _m->buffer[0] - 1,
                                             _DMS_HILITE );
                            ShowMouse();

                            if( _m != cm ) {
                                cm = _m;
                                if( _drawmenu( cm, 0 ) == FALSE )
                                    return MENUNULL;
                            }

                            keepon = 0;
                            while( ie->mousey == 0 && ie->buttons & MBUTTON_LEFT ) {
                                ie = GetInput();
                                if( ( ie->mousex < _m->buffer[0] ||
                                    ie->mousex > _m->buffer[1] ) ) {
                                    /* select another menu */
                                    keepon = 1;
                                    break;
                                }
                            }

                            HideMouse();
                            _drawmenustring( _m->MenuName, 0,
                                             _m->buffer[0]+1, 0,
                                             _m->buffer[1] - _m->buffer[0] - 1,
                                             _DMS_NORMAL );
                            ShowMouse();

                            break;
                        }
                    }
                }
                if( _m == NULL )
                {
                    if( cm == NULL && !keepon ) {
                        /* exit if no valid menu found and no menu is open */
                        cancel = TRUE;
                        goto done;
                    }

                    keepon = 1;
                }
                if( keepon ) {
                    if( cm ) {
                        _destroymenu( cm );
                        cm = NULL;
                    }
                }
                ie = GetInput();

            }
            if( !cm )
                goto done;
        }

        ie = _getmenu( m, cm, 0, 0 );
        if( ie->key )
        {
            switch( ie->key )
            {
                case BK_RIGHT:  _destroymenu( cm );
                                if( cm->Next )
                                {
                                    cm = cm->Next;
                                }
                                else
                                {
                                    cm = m;
                                }
                                if( _drawmenu( cm, 0 ) == FALSE )
                                    return MENUCANCEL;

                                break;
                case BK_LEFT:   _destroymenu( cm );
                                /* get previous menu */
                                for( _m = m;(_m->Next != cm) && _m->Next; _m = _m->Next );
                                cm = _m;
                                if( _drawmenu( cm, 0 ) == FALSE )
                                    return MENUCANCEL;
                                break;
                case BK_ESC:    cancel = TRUE;
                case BK_RETURN: goto done;
                default:        /* look for Accelerator */
                                for( _m = m; _m; _m = _m->Next )
                                {
                                    if( mi = _findshortkey( _m, ie->key ) ) {
                                        accelerator = ACCELERATORBIT;
                                        curr = mi;
                                        goto done;
                                    }
                                }

                                /* look if menu selected */
                                _m = _findmenu( m, ie->key );
                                if( _m != NULL ) {

                                    _destroymenu( cm );
                                    cm = _m;
                                    if( _drawmenu( cm, 0 ) == FALSE )
                                        return MENUCANCEL;
                                    break;
                                }

                                /* look for hot key */
                                if( mi = _findhotkey( cm, ie->key ) ) {
                                    curr = mi;
                                    goto done;
                                }

            }
        } else if( curr == NULL )
            break;
    }
done:
    if( cm )
        _destroymenu( cm );

    SetCursorSize( sav.Start, sav.End );

    if( cancel == FALSE && curr != NULL )
        return (_menucode( m, cm, curr )|accelerator);
    return MENUCANCEL;
}


static struct MenuItem *_findshortkey( struct Menu *m, unsigned int key )
{
    struct MenuItem *mi;

    for( mi = m->First; (mi->ShortKey != key) && mi; mi = mi->Next );
    return mi;
}

static struct MenuItem *_findhotkey( struct Menu *m, unsigned int key )
{
    struct MenuItem *mi;
    char *cp;

    for( mi = m->First; mi; mi = mi->Next )
    {
        if( mi->Flags & ( ITEM_SECTION | ITEM_DISABLE ) )
            continue;

        cp = strrchr( mi->ItemName, (int)MENUHOTINTRODUCER );
        if( !cp || _2lower( (char)(key) ) != _2lower( cp[1] ) )
            continue;

        break;
    }
    return mi;
}


static int _drawmenustring( str, skey, x, y, barwidth, flag )
char *str;
unsigned skey;
int x, y, barwidth, flag;
{
    char keyname[20];
    char *cp;
    int len, length, xoffset;
    unsigned int hiliteattr, attr;
    cp = strrchr( str, (int)MENUHOTINTRODUCER );
    length = strlen( str );

    xoffset = x;

    if( flag == _DMS_DISABLE ) {
        attr = hiliteattr = CR_MENUDISABLE;
    } else if( flag == _DMS_HILITE ) {
        attr = CR_MENUBAR;
        hiliteattr = CR_MENUBARHOT;
    } else {
        attr = CR_MENU;
        hiliteattr = CR_MENUHOTKEY;
    }

    HideMouse();
    _printchar( ' ', barwidth, attr, y ? x - 1 : x, y );

    if( cp ) {
        --length;
        len = cp - str;
        _nprint( str, len, attr, x, y );
        x += len;
        VideoOut( hiliteattr, *++cp, x++, y );
        ++cp;
    } else
        cp = str;

    VideoStr( attr, cp, x, y );

    if( skey ) {
        e_getkeyname( skey, keyname );
        x = xoffset + barwidth - strlen( keyname ) - 2;
        VideoStr( attr, keyname, x, y );
        ShowMouse();
        return(barwidth - 2);
    }

    ShowMouse();
    return length;
}


static unsigned long _menucode( first, menu, item )
struct Menu *first, *menu;
struct MenuItem *item;
{
    struct Menu *m;
    struct MenuItem *mi;
    unsigned int i, j;

    for( i = 1, m = first; m; m = m->Next, ++i ) {
        for( j = 1, mi = m->First; mi; mi = mi->Next, ++j ) {
            if( mi == item ) {
                return SHIFTMENU(i) | SHIFTITEM(j);
                break;
            }
        }
    }

    return 0L;
}


static BOOL _drawmenu( struct Menu *m, int top ) {

    struct Window *w;
    struct MenuItem *mi;
    struct InputEvent *ie;
    int maxwidth = 0, numits = 0, xoff = 0;
    char keyname[20];
    struct _intuitionbase savibase;

    if( ! ( w = malloc( sizeof( struct Window ) ) ) )
        return FALSE;

    if( menuhandler )
        (*menuhandler)( m, NULL, MH_OPENMENU );

    m->Window = w;
    m->Top = top;
    for( mi = m->First; mi; mi = mi->Next ) {
        mi->Top = ++xoff;
        ++numits;
        if( mi->Flags & ITEM_SECTION )
            continue;
        e_getkeyname( mi->ShortKey, keyname );
        maxwidth = _MAX( maxwidth, strlen( mi->ItemName ) +
            strlen( keyname ) + 1 );
    }
    m->Width = maxwidth + 4;
    m->Height = numits + 2;

    w->NewLeft = m->buffer[0] - 1;
    w->NewTop = 1 + top;
    w->Width = m->Width;
    w->Height = m->Height;
    w->Flags = 0L;
    w->Title = NULL;

    savibase = ibase;
    ibase.WindowColor = ibase.BorderColor = CR_MENU;
    if( !OpenWindow( w, 0 ) ) {
        ibase = savibase;
        return FALSE;
    }
    ibase = savibase;

    HideMouse();

    if( ! top )
    {
        VideoOut( CR_MENU, '', m->buffer[0], 0 );
        VideoOut( CR_MENU, '', m->buffer[0], 1 );
        VideoOut( CR_MENU, '', m->buffer[1], 0 );
        VideoOut( CR_MENU, '', m->buffer[1], 1 );
    }
    for( mi = m->First; mi; mi = mi->Next )
    {
        if( mi->Flags & ITEM_SECTION )
        {
            VideoOut( CR_MENU, '', w->Left , mi->Top + w->Top );
            _printchar( '', m->Width - 2,  CR_MENU, w->Left + 1, mi->Top + w->Top );
            VideoOut( CR_MENU, '', w->Left + w->Width - 1, mi->Top + w->Top );
        }
        else if( mi->Flags & ITEM_DISABLE )
        {
            _drawmenustring( mi->ItemName, mi->ShortKey, w->Left + 2,
                mi->Top + w->Top, w->Width-2, _DMS_DISABLE );
        }
        else
        {
            _drawmenustring( mi->ItemName, mi->ShortKey, w->Left + 2,
                mi->Top + w->Top, w->Width-2, _DMS_NORMAL );
        }
    }

    ShowMouse();
    return TRUE;
}


static void _destroymenu( struct Menu *m ) {

    if( menuhandler )
        (*menuhandler)( m, NULL, MH_DESTROYMENU );

    HideMouse();
    if( ! m->Top )
    {
        VideoOut( CR_MENU, ' ', m->buffer[0], 0 );
        VideoOut( CR_MENU, ' ', m->buffer[1], 0 );
    }
    ShowMouse();
    CloseWindow( m->Window );
    free( m->Window );
}


static struct Menu *_findmenu( struct Menu *m, int key )
{
    char *cp;
    char c = _altname( key );

    for( ; m; m = m->Next )
    {
                // scan m->MenuName for hot key
        cp = strrchr( m->MenuName, (int)MENUHOTINTRODUCER );
        if( cp && c == _2lower( cp[1] ) )
        {
            return( m );
        }
    }
    return( NULL );
}


static char _altname( unsigned int key )
{
    static char keytable[] = "qwertyuiop    asdfghjkl     zxcvbnm";
    static char numtable[] = "1234567890-=";

    int menukey;
    char *cp;

    if( (char)key != '\0' )
    {
        return '\0';
    }

    key = ( key >> 8 );

    if( key >= 120 && key <= 131 )
        return numtable[key-120];

    key -= 16;
    if( key > (sizeof( keytable ) / sizeof( char )) ) {
        return '\0';
    }

    return keytable[key];
}


// #pragma optimize( "egl", off )
static struct InputEvent *_getmenu( first, am, top, left )
struct Menu *first, *am;
int top, left;
{
    struct Menu *m, *_m;
    struct MenuItem *mi;
    struct InputEvent *ie;

    ie = ibase.ie;

    for( mi = am->First; mi && (mi->Flags & (ITEM_SECTION | ITEM_DISABLE));
         mi = mi->Next );

    for( m = am;; )
    {
        if( ie->key || !(ie->buttons & MBUTTON_LEFT) )
        {
            if( menuhandler )
                (*menuhandler)( m, mi, MH_CHECKITEM );

            if( mi )
                _drawmenustring( mi->ItemName, mi->ShortKey,
                am->Window->Left + 2, mi->Top + am->Window->Top,
                am->Window->Width - 2, _DMS_HILITE );

            ie = ReadInput();

            if( mi )
                _drawmenustring( mi->ItemName, mi->ShortKey,
                am->Window->Left + 2, mi->Top + am->Window->Top,
                am->Window->Width-2, _DMS_NORMAL );

        } else if( ie->buttons & MBUTTON_LEFT ) {
            do {
                ie = GetInput();
                if( ie->key ) break;
                if( ie->mousey == 0 ) {
                    /* get another menu */
                    curr = (struct MenuItem *)1;
                    return ie;
                }
            } while( ie->mousey <= 1 && ie->buttons & MBUTTON_LEFT);
        }

        if( ie->key )
        {
            switch( ie->key )
            {
                case BK_DOWN:   if( ! mi )
                                    break;
                                do
                                {
                                    mi = mi->Next;
                                }while( ( mi->Flags & (ITEM_SECTION | ITEM_DISABLE) ) && mi );
                                if( ! mi )
                                {
                                    for( mi = m->First;
                                      mi && (mi->Flags & (ITEM_SECTION | ITEM_DISABLE));
                                      mi = mi->Next );
                                }
                                break;
                case BK_UP:     if( ! mi )
                                    break;
                                {
                                    struct MenuItem *mmi, *lastmi;
                                    for( lastmi = NULL, mmi = m->First; mmi ; mmi = mmi->Next )
                                    {
                                        if( mmi == mi && lastmi )
                                            break;

                                        if( ! ( mmi->Flags & (ITEM_SECTION | ITEM_DISABLE) ) )
                                            lastmi = mmi;
                                    }
                                    mi = lastmi;
                                }
                                break;
                case BK_ESC:    curr = NULL;
                                return( ie );

                case BK_RETURN: curr = mi;
                default:        return( ie );
            }
        }
        if( ie->buttons )
        {
            int i;
            struct MenuItem *mmi;

            if( ie->mousex < am->Window->Left || ie->mousex
                >= am->Window->Left + am->Window->Width
                || ie->mousey < am->Window->Top
                || ie->mousey >= am->Window->Top + am->Window->Height )
            {                                          /* outside window */
                if( ie->mousey != 0 )
                {
                    curr = NULL;    /* mark NULL to finish CheckMenu */
                }
                else
                {
                    if( ie->buttons )
                        curr = (struct MenuItem *)1; /* mark to get next Menu */
                    else
                        curr = NULL;
                }

                return( ie );
            }

            if( ( ie->mousey != am->Window->Top )
             && ( ie->mousey != am->Window->Top + am->Window->Height - 1 )
             && ( ie->mousey > am->Window->Top )
             && ( mi ) )
                                                      /* inside window */
            {
                for( i = ie->mousey - 2 - top, mmi = am->First;
                      i > 0; --i, mmi = mmi->Next );

                if( mmi->Flags & ( ITEM_SECTION | ITEM_DISABLE ) )
                {
                    continue;
                }

                if( menuhandler )
                    (*menuhandler)( m, mmi, MH_CHECKITEM );
                _drawmenustring( mmi->ItemName, mmi->ShortKey,
                                 am->Window->Left + 2,
                                 mmi->Top + am->Window->Top,
                                 am->Window->Width-2, _DMS_HILITE );

                curr = mi = mmi;

                // wait for button release
                do
                {
                    curr = mi = mmi;

                    ie = GetInput();

                    if( ie->mousey == 0 ) {
                        _drawmenustring( mi->ItemName, mi->ShortKey,
                                     am->Window->Left + 2,
                                     mi->Top + am->Window->Top,
                                     am->Window->Width-2, _DMS_NORMAL );

                        // just mark to open next menu
                        curr = (struct MenuItem *)1;
                        return ie;
                    }
                        // pointer on top of window
                    if( ie->mousey == 1 + top )
                        continue;

                        // find item, if any
                    for( i = ie->mousey - 2 - top, mmi = am->First;
                        i > 0 && mmi; --i, mmi = mmi->Next );

                           // same item and not on title line
                    if( mmi == curr && ie->mousey != 0 &&
                        !( ie->mousex < am->Window->Left ||
                            ie->mousex >= am->Window->Left + am->Window->Width ))
                        continue;

                        // old menu item is hilighted, draw it normal
                    if( mi && !(mi->Flags & ( ITEM_SECTION | ITEM_DISABLE ) ) )
                        _drawmenustring( mi->ItemName, mi->ShortKey,
                                     am->Window->Left + 2,
                                     mi->Top + am->Window->Top,
                                     am->Window->Width-2, _DMS_NORMAL );

                            // no valid item and top on window
                    if( !mmi && ie->mousey <= top ) {
                        return ie;
                    }
                            // no valid item or not selectable item
                    if( !mmi || ( mmi->Flags & ( ITEM_SECTION | ITEM_DISABLE ) ) ) {
                        mmi = mi = NULL;
                        continue;
                    }
                    if( ie->mousey > am->Window->Top + am->Window->Height ) {
                        mmi = mi = NULL;
                        continue;
                    }
                    if( ie->mousex <  am->Window->Left ||
                        ie->mousex >= am->Window->Left + am->Window->Width ) {
                        mmi = mi = NULL;
                        continue;
                    }

                    if( menuhandler )
                        (*menuhandler)( m, mmi, MH_CHECKITEM );
                    _drawmenustring( mmi->ItemName, mmi->ShortKey,
                                 am->Window->Left + 2,
                                 mmi->Top + am->Window->Top,
                                 am->Window->Width-2, _DMS_HILITE );

                }while( ie->buttons & MBUTTON_LEFT );

                    // reset the button info, so CheckMenu() recognizes it
                ie->buttons = MBUTTON_LEFT;

                return( ie );
            }
        }
    }
}
// #pragma optimize( "", on )


/*
 *      NAME:           Beep
 *      ARGUMENTS:      
 *      RETURN TYPE:    
 *
 *      Purpose:        Sonds the bell
 *
 * (c) M. Schollmeyer 1990,1991
 */

void Beep( void )
{
    Sound( ibase.soundfreq, ibase.sounddur );
}



static int _msgbox( unsigned, int, int, char *, char *, va_list );

/*
 *      NAME:           DoErrorBox
 *      ARGUMENTS:      
 *      RETURN TYPE:    
 *
 *      Purpose:        
 *
 * (c) M. Schollmeyer 1990,1991
 */
void DoErrorBox( int topic, char *fmt, ... ) {

    va_list marker;

    va_start( marker, fmt );
    _msgbox( DMB_OK|DMB_BEEP, topic, -1, "Error", fmt, marker );
}


/*
 *      NAME:           DoMessageBox
 *      ARGUMENTS:      
 *      RETURN TYPE:    
 *
 *      Purpose:        
 *
 * (c) M. Schollmeyer 1990,1991
 */
int DoMessageBox( flags, topic, caption, fmt, ...)
int flags;
int topic;
char *caption, *fmt;
{
    va_list marker;

    va_start( marker, fmt );
    return _msgbox( flags, topic, -1, caption, fmt, marker );
}


/*
 *      NAME:           DoYMessageBox
 *      ARGUMENTS:      
 *      RETURN TYPE:    
 *
 *      Purpose:        
 *
 * (c) M. Schollmeyer 1990,1991
 */
int DoYMessageBox( flags, topic, wy, caption, fmt, ... )
int flags;
int topic;
int wy;
char *caption;
char *fmt;
{
    va_list marker;

    va_start( marker, fmt );
    return _msgbox( flags, topic, -1, caption, fmt, marker );
}


/*
 *    Name: _msgbox
 *  Return: DMB_xxx (user response)
 * Purpose: Opens a window and displays a message, then waits for
 *          user response.
 *
 *
 * (c)M.Schollmeyer
 */
// #pragma optimize( "elg", off )
#pragma check_stack( on )
static int _msgbox( unsigned flags, int topic, int wy,
char *caption, char *fmt, va_list marker ) {

    static char *itstr[] =
        { "&OK", "&Yes", "&No", "&Retry", "Cancel ", "&Help" };

    char buffer[250];
    int scrbuffer[11 * SCREEN_WIDTH];

    static struct Window win = { -1, 0, 0, 0, 0, 0, NULL, 0L, NULL };

    struct Gadget gad[6];
    struct Gadget *g, *first, *f;
    struct IntuiText itgad[6];
    char *cp;

    unsigned int i;
    int width, height, len, xoff, row;
    struct _intuitionbase savibase;

    /* First, make a beep */
    if( flags & DMB_BEEP ) {
        Beep();
        CLEARBIT( flags, DMB_BEEP );
    }

    /* Setup window */

    if( flags & DMB_STATIC ) {
        win.Buffer = (void _far *)scrbuffer;
        CLEARBIT( flags, DMB_STATIC );
        SETBIT( win.Flags, STATIC_WINDOW );
    } else {
        win.Buffer = NULL;
        CLEARBIT( win.Flags, STATIC_WINDOW );
    }

    /* make the display string */
    vsprintf( buffer, fmt, marker );

    for( cp = buffer, width = height = 0; *cp ; ) {
        for( i = 0; *cp && *cp != '\n'; ++i, ++cp );
        width = _MAX( width, i );
        while( *cp == '\n' ) {
            ++height;
            ++cp;
        }
    }
    height += 8;
    width += 4;

    win.NewTop = wy;

    win.Width = _MAX( width, strlen( caption ) + 4 );
    win.Height = height;
    win.Title = caption;

    /* setup gadgets */
    for( i = 0; i < 6; ++i ) {
        itgad[i].Text = NULL;
        itgad[i].TextBorder = NULL;
        itgad[i].Left = 1;
        itgad[i].Top = 0;
        itgad[i].Text = itstr[i];
        itgad[i].Flags = 0L;
        itgad[i].Width = 0;

        gad[i].Next     = NULL;
        gad[i].IText    = &itgad[i];
        gad[i].Top      = win.Height-4;
        gad[i].Width    = strlen(itgad[i].Text) + 3;
        gad[i].Height   = 3;
        gad[i].Flags    = BOOLGADGET|ENDGADGET;
    }

    width = 1;
    g = first = NULL;

    flags &= 0x1f;

    for( i = 0; flags & 0x1f; ++i ) {
        if( flags & 1 ) {
            if( g ) g->Next = &gad[i];
            g = &gad[i];
            if( !first) first = g;
            g->Left = width;
            width += g->Width + 1;
        }
        flags = flags >> 1;
    }
    if( first && first != &gad[4] ) {
        ++first->IText->Text;
        SETBIT( first->Flags, ACTIVATED );
    }

    if( topic ) {
        if( g ) g->Next = &gad[5];
        width += gad[5].Width + 2;
        win.Width = _MAX( win.Width, width );
        gad[5].Left = win.Width-9;
    } else
        win.Width = _MAX( win.Width, width );

    /* if there is more than one gadget: shift gadgets right until they
       are balanced */
    width = win.Width - width;
    if( first->Next ) {
        for( f = first->Next; width > 0; f = f->Next ) {
            if( f == &gad[5] || !f )
                f = first->Next;
            for( g = f; g && g != &gad[5]; g = g->Next )
                ++(g->Left);
            --width;
        }
    }

    SETBIT( first->Flags, ACTIVATED );

    savibase = ibase;
    ibase.WindowColor = CR_ERRORWINDOW;
    ibase.GadgetColor = CR_ERRORGAD;
    ibase.GadgetSelect = CR_ERRORGADSELECT;
    ibase.GadgetHot = CR_ERRORGADHOT;
    ibase.GadgetHotSelect = CR_ERRORGADSELHOT;
    ibase.BorderColor = CR_ERRORWINDOW;

    if( !OpenWindow( &win, topic ) ) {
        ibase = savibase;
        return DMB_CANCEL;
    }

    for( cp = buffer, row = 1; cp[0] ; ) {
        for( len = 0; cp[len] && cp[len] != '\n'; ++len );

        xoff = (win.Width-2-len)/2;

        _printl( cp, len, CR_ERRORWINDOW,
            win.Left+1+xoff, row+win.Top+1 );

        cp += len;
        while( *cp == '\n' ) {
            ++cp;
            ++row;
        }
    }

    do {
        g = OpenGadget( first, win.Left, win.Top, 0L );
        if( g == &gad[5] )
            help( topic );
    } while( g == &gad[5] );

    CloseWindow( &win );

    ibase = savibase;

    if( !g ) return DMB_CANCEL;
    for( i = 0, flags = 1; g != &gad[i]; ++i ) flags = flags << 1;
    return flags;
}
#pragma check_stack()
// #pragma optimize( "", on )


/*
 *    Name: PutError
 *  Return: DMB_xxx (user response)
 * Purpose: opens a dialog box with the latest DOS error.
 *
 *
 *
 * (c)M.Schollmeyer
 */
int PutError( int err, char *file ) {

    struct DOSError *de;               /* structure to carry DOS error message */
    char buffer[SCREEN_WIDTH];         /* general purpose buffer */
    static char *fmtnofile = "%s";     /* format strings */
    static char *fmtname = "%s\n'%s'";
    char *fmt;

    if( file ) {
        /* file name is given: make it short and take 'fmtname' as format */
        MakeShortName( buffer, file, SCREEN_WIDTH-7 );
        fmt = fmtname;
    } else {
        /* no filename given, take 'fmtnofile' as format */
        buffer[0] = 0;
        fmt = fmtnofile;
    }

    /* get the error message from DOS */
    de = DOSError( err );

    if( de->Code )
        return DoMessageBox( de->Flags|DMB_STATIC, 0, "DOS Error",
                             fmt, de->Msg, buffer );
    else
        return DoMessageBox( de->Flags|DMB_STATIC, 0, "DOS Error",
                             "File '%s'", buffer );
}


/* end of file intui.c */
