#ifdef __TURBOC__
#pragma inline
#endif
/*  <- declarations */
/*
 * These routines decode GIF files and produce Windows metafiles.
 * They were based on giftest.c by Jonathan O'Neal, which in turn
 * is based on decode.c by Steven A. Bennett (see the copyright
 * notice below).
 *
 * Adapted by Hippocrates Sendoukas, Athens, March 1994
 *
 ********** Copyright notice for decode.c ************
 *
 * DECODE.C - An LZW decoder for GIF
 * Copyright (C) 1987, by Steven A. Bennett
 *
 * Permission is given by the author to freely redistribute and include
 * this code in any program as long as this credit is given where due.
 *
 * In accordance with the above, I want to credit Steve Wilhite who wrote
 * the code which this is heavily inspired by...
 *
 * GIF and 'Graphics Interchange Format' are trademarks (tm) of
 * Compuserve, Incorporated, an H&R Block Company.
 */
#define	FILTNAME			"gif"
#include "common1.c"

#define MAX_CODES			4095

/* Static variables */
static Int16 curr_size,                     /* The current code size */
             clear,                         /* Value for a clear code */
             ending,                        /* Value for a ending code */
             newcodes,                      /* First available code */
             top_slot,                      /* Highest code for current size */
             slot;                          /* Last read code */

/*
 * The following static variables are used
 * for seperating out codes
 */
static int   navail_bytes = 0,              /* # bytes left in block */
             nbits_left = 0;                /* # bits left in current byte */
static Uns8	b1,                           /* Current byte */
		byte_buff[257];               /* Current block */

struct gifhdr
{
  unsigned	nx,ny,bpp,ncolors,interlaced,maxdy,wpl,firsty;
  HPALETTE	hpal;
  HDC		hdc;
  HANDLE	hmf;
  Uns8		*dibbits,*gifbits,*dibbitptr;
};

/*  -> declarations */
/*  <- adjust_bitloc */

static void PASCAL near
adjust_bitloc(struct gifhdr *gs,unsigned y)
{
  unsigned dy,y2;

  y2 = y + gs->maxdy;
  if (y2>gs->ny) y2 = gs->ny;
  dy = y2 - y;
  gs->dibbitptr = gs->dibbits + (dy-1)*2*gs->wpl;
}

/*  -> adjust_bitloc */
/*  <- init_meta */

static int PASCAL NEAR
init_meta(struct gifhdr *gs)
{
  unsigned	nx,ny,maxdy;
  unsigned long	temp;

  nx = gs->nx;
  ny = gs->ny;
  temp = (unsigned long) nx * (unsigned long) gs->bpp;
  temp += 31;
  temp /= 32;
  temp *= 2;
  gs->wpl = (unsigned) temp;
  maxdy = 16384u/gs->wpl;                           /* 32K chunks */
  if (maxdy==0) return IE_TOO_BIG;
  if (maxdy>1) maxdy--;
  if (maxdy>ny) maxdy = ny;
  gs->maxdy = maxdy;
  if ( (gs->dibbits=malloc(2*gs->wpl*maxdy)) == NULL	||
       (gs->gifbits=malloc(nx)) == NULL		||
       (gs->hdc=CreateMetaFile(NULL)) == 0		)
    return IE_MEM_FULL;

  adjust_bitloc(gs,0);

  SetWindowOrgEx(gs->hdc,0,0,NULL);
  SetWindowExtEx(gs->hdc,nx,ny,NULL);
  if (gs->hpal)
    {
      SelectPalette(gs->hdc,gs->hpal,0);
      RealizePalette(gs->hdc);
    }
  return IE_OK;
}

/*  -> init_meta */
/*  <- finish_meta */

static void PASCAL NEAR
finish_meta(struct gifhdr *gs,int retcode)
{
  if (gs->gifbits) free(gs->gifbits);
  if (gs->dibbits) free(gs->dibbits);
  if (gs->hdc)
    {
      gs->hmf = CloseMetaFile(gs->hdc);
      if (gs->hmf && retcode!=IE_OK)
        {
          DeleteMetaFile(gs->hmf);
          gs->hmf = 0;
        }
    }
}

/*  -> finish_meta */
/*  <- comb8to1 */

static void PASCAL NEAR
comb8to1(Uns8 *dibptr,Uns8 *gifptr,unsigned nx)
{
  unsigned chunks,rem,i,j,k;
  static Uns8 mask[8] = { 128,64,32,16,8,4,2,1 };

  chunks = nx/8;
  rem = nx % 8;
  for (i=j=0; i<chunks; i++)
    {
      dibptr[i] = '\0';
      for (k=0; k<8; k++,j++)
        if (gifptr[j])  dibptr[i] |= mask[k];
    }
  dibptr[i] = '\0';
  for (k=0; k<rem; k++,j++)
    if (gifptr[j]) dibptr[i] |= mask[k];
}

/*  -> comb8to1 */
/*  <- comb2to1 */

static void PASCAL NEAR
comb2to1(Uns8 *dibptr,Uns8 *gifptr,unsigned nx)
{
#ifdef __TURBOC__

#ifdef __WIN32__
	asm	mov	ebx,nx
	asm	mov	esi,gifptr
	asm	mov	edi,dibptr
	asm	mov	ecx,ebx
	asm	shr	ecx,1
	asm	jz	NOPE1
TOP1:	asm	lodsb
	asm	shl	al,4
	asm	mov	ah,al
	asm	lodsb
	asm	or	al,ah
	asm	stosb
	asm	loop	TOP1
NOPE1:	asm	and	ebx,1
	asm	jz	NOPE2
	asm	lodsb
	asm	shl	al,4
	asm	stosb
NOPE2:	;
#else
	asm	push	ds
	asm	mov	bx,nx
	asm	lds	si,gifptr
	asm	les	di,dibptr
	asm	mov	cx,bx
	asm	shr	cx,1
	asm	jz	NOPE1
TOP1:	asm	lodsb
	asm	shl	al,4
	asm	mov	ah,al
	asm	lodsb
	asm	or	al,ah
	asm	stosb
	asm	loop	TOP1
NOPE1:	asm	and	bx,1
	asm	jz	NOPE2
	asm	lodsb
	asm	shl	al,4
	asm	stosb
NOPE2:	asm	pop	ds
#endif

#else
  unsigned chunks,i,j,hi;

  chunks = nx/2;
  for (i=j=0; i<chunks; i++)
    {
      hi = gifptr[j++];
      dibptr[i] = (hi << 4) | gifptr[j++];
    }
  if (nx & 1) dibptr[i] = gifptr[j] << 4;
#endif
}

/*  -> comb2to1 */
/*  <- out_line */

static void PASCAL NEAR
out_line(struct gifhdr *gs,unsigned y,BITMAPINFO *bi,DWORD mode)
{
  unsigned	nx,wpl,dy;

  if (y>=gs->ny) return;
  nx = gs->nx;
  switch (gs->bpp)
    {
      case 1:
        comb8to1(gs->dibbitptr,gs->gifbits,nx);
        break;
      case 4:
        comb2to1(gs->dibbitptr,gs->gifbits,nx);
        break;
      case 8:
        memcpy(gs->dibbitptr,gs->gifbits,nx);
    }
  gs->dibbitptr -= 2*gs->wpl;
  dy = y - gs->firsty + 1;

  if (y==gs->ny-1 || dy>=gs->maxdy)
    {
      wpl = gs->wpl;
      bi->bmiHeader.biHeight = dy;
      bi->bmiHeader.biSizeImage = 2*wpl*dy;
      StretchDIBits(gs->hdc,0,gs->firsty,nx,dy,0,0,nx,dy,
                    (LPSTR)gs->dibbits,bi,DIB_RGB_COLORS,mode);
      adjust_bitloc(gs,y+1);
      gs->firsty = y+1;
    }
}

/*  -> out_line */
/*  <- init_exp */

/*
 * This function initializes the decoder for reading a new image.
 */
static void PASCAL NEAR
init_exp(Int16 size)
{
  curr_size = (Int16)(size + 1);
  top_slot = (Int16)(1 << curr_size);
  clear = (Int16)(1 << size);
  ending = (Int16)(clear + 1);
  slot = newcodes = (Int16)(ending + 1);
  navail_bytes = nbits_left = 0;
}

/*  -> init_exp */
/*  <- header */

/*
 * Read the GIF header
 */
static int PASCAL NEAR
header(FILE *f,struct gifhdr *gs,RGBQUAD *q)
{
  int		x;
  unsigned	PixelWidth,p;
  Uns8		buf[16];
  static RGBQUAD mdef[2] =
    { { 0x00,	0x00,	0x00,	0 },
      { 0xff,	0xff,	0xff,	0 }  };

  buf[0] = '\0';
  fread(buf,1,6,f);
  if (memcmp(buf,"GIF",3)) /* don't care about 87a or 89a or 93a... */
    return IE_BAD_FILE_DATA;

  gs->nx = readUns16(f);			/* Get X resolution */
  gs->ny = readUns16(f);			/* Get Y resolution */
  p = getc(f);				/* Get pixel width & flags */
  PixelWidth = (p & 7) + 1;
  gs->bpp = PixelWidth;

  if (PixelWidth!=1 && PixelWidth!=4 && PixelWidth!=8)
    return IE_BAD_FILE_DATA;

  gs->ncolors = 1 << PixelWidth;
  memcpy(q,mdef,2*sizeof(mdef[0]));

  getc(f);				/* read Background */
  getc(f);				/* read <Reserved> */

  if (p & 0x80)				/* global color map */
    for (x=gs->ncolors; x>0; q++,x--)
      {
        q->rgbRed	= readUns8(f);
        q->rgbGreen	= readUns8(f);
        q->rgbBlue	= readUns8(f);
        q->rgbReserved	= 0;
      }

  while ( (x=getc(f)) == '!' )		/* get ',' or '!' */
    {					/* if it is ! then GIF extension */
      getc(f);				/* extended function code */
      for (;;)
        {
          x = getc(f);			/* byte count */
          if (x<0) return IE_BAD_FILE_DATA;
          if (x==0) break;
          while (x--) getc(f);		/* extended function data */
        }
    }

  readUns16(f);				/* left offset */
  readUns16(f);				/* top offset */
  readUns16(f);				/* width */
  readUns16(f);				/* height */

  x = getc(f);				/* interlaced?  local map? */
  gs->interlaced = ( (x & 0x40) != 0 );
  if (gs->interlaced) return IE_UNSUPP_VERSION;


  if (x & 0x80)				/* get local color map */
    {
      x = 1 << ((x & 7) + 1);
      while (x--)
        {
          getc(f);			/* read color data (RGB) */
          getc(f);
          getc(f);
        }
    }

  return feof(f) ? IE_BAD_FILE_DATA : IE_OK;
}

/*  -> header */
/*  <- get_next_code */

/*
 * gets the next code from the GIF file.  Returns the code, or else
 * zero in case of file errors...
 */
static unsigned PASCAL NEAR
get_next_code(FILE *f)
{
  unsigned ret;
  static unsigned code_mask[13] = {
     0,
     0x0001, 0x0003,
     0x0007, 0x000F,
     0x001F, 0x003F,
     0x007F, 0x00FF,
     0x01FF, 0x03FF,
     0x07FF, 0x0FFF
     };
  static Uns8 *pbytes;                 /* Pointer to next byte in block */

  if (nbits_left == 0)
    {
      if (navail_bytes <= 0)
        {
          /*
           * Out of bytes in current block, so read next block
           */
          pbytes = byte_buff;
          if ( (navail_bytes=getc(f)) <= 0 ||
               fread(byte_buff,1,navail_bytes,f)!=navail_bytes )
            return 0;
        }
      b1 = *pbytes++;
      nbits_left = 8;
      --navail_bytes;
    }

  ret = b1 >> (8 - nbits_left);
  while (curr_size > nbits_left)
    {
      if (navail_bytes <= 0)
        {
          /*
           * Out of bytes in current block, so read next block
           */
          pbytes = byte_buff;
          if ( (navail_bytes=getc(f)) <= 0 ||
               fread(byte_buff,1,navail_bytes,f)!=navail_bytes )
            return 0;
        }
      b1 = *pbytes++;
      ret |= b1 << nbits_left;
      nbits_left += 8;
      --navail_bytes;
    }
  nbits_left -= curr_size;
  return (ret & code_mask[curr_size]);
}

/*  -> get_next_code */
/*  <- decoder */

/*
 * This function decodes an LZW image, according to the method used
 * in the GIF spec.  The function gets its codes from get_next_code()
 * which is responsible for reading blocks of data and separating them
 * into the proper size codes.
 *
 * Returns: 0 if successful, else negative.
 */
static int PASCAL NEAR
decoder(FILE *f,struct gifhdr *gs,BITMAPINFO *bi,DWORD mode)
{
  Uns8 *sp,*bufptr;
  Int16 code,fc,oc,c,size;
  int	linecount,retcode = IE_OK,bufcnt;
  Uns8	*stack,				/* Stack for storing pixels */
  	*suffix;			/* Suffix table */
  Uns16	*prefix;			/* Prefix linked list */

  if ( (prefix = malloc((MAX_CODES+1)*sizeof(Uns16) +
                        2*(MAX_CODES+1)*sizeof(Uns8))) == NULL )
    return IE_MEM_FULL;
  stack = (Uns8 *) (&prefix[MAX_CODES+1]);
  suffix = stack + MAX_CODES+1;
  /*
   * Initialize for decoding a new image...
   */
  if ( (size = (Int16)getc(f)) < 0 )       /* get initial code size */
    { retcode = IE_BAD_FILE_DATA; goto GETOUT; }
  if (size < 2 || 9 < size)
    { retcode = IE_BAD_FILE_DATA; goto GETOUT; }
  init_exp(size);

  /*
   * Initialize in case they forgot to put in a clear code.
   * (This shouldn't happen, but we'll try and decode it anyway...)
   */
  oc = fc = 0;
  linecount = 0;

  /*
   * Set up the stack pointer and decode buffer pointer
   */
  sp = stack;
  bufptr = gs->gifbits;
  bufcnt = gs->nx;

  /*
   * This is the main loop.  For each code we get we pass through the
   * linked list of prefix codes, pushing the corresponding "character" for
   * each code onto the stack.  When the list reaches a single "character"
   * we push that on the stack too, and then start unstacking each
   * character for output in the correct order.  Special handling is
   * included for the clear code, and the whole thing ends when we get
   * an ending code.
   */
  while ((c = (Int16) get_next_code(f)) != ending)
    {
      /*
       * If we had a file error, return without completing the decode
       */
      if (feof(f)) { retcode = IE_BAD_FILE_DATA; goto GETOUT; }

      /*
       * If the code is a clear code, reinitialize all necessary items.
       */
      if (c == clear)
        {
          curr_size = (Int16)(size + 1);
          slot = newcodes;
          top_slot = (Int16)(1 << curr_size);

          /*
           * Continue reading codes until we get a non-clear code
           * (Another unlikely, but possible case...)
           */
          while ( (c=(Int16)get_next_code(f)) == clear)
            ;
          if (feof(f)) { retcode = IE_BAD_FILE_DATA; goto GETOUT; }

          /*
           * If we get an ending code immediately after a clear code
           * (Yet another unlikely case), then break out of the loop.
           */
          if (c == ending) break;

          /*
           * Finally, if the code is beyond the range of already set codes,
           * (This one had better NOT happen...  I have no idea what will
           * result from this, but I doubt it will look good...) then set it
           * to color zero.
           */
          if (c >= slot)  c = 0;

          oc = fc = c;

          /*
           * And let us not forget to put the char into the buffer... And
           * if, on the off chance, we were exactly one pixel from the end
           * of the line, we have to send the buffer to the out_line()
           * routine...
           */
          *bufptr++ = (Uns8) c;
          if (--bufcnt == 0)
            {
              out_line(gs,linecount++,bi,mode);
              bufptr = gs->gifbits;
              bufcnt = gs->nx;
            }
        }
      else
        {
          /*
           * In this case, it's not a clear code or an ending code, so
           * it must be a code code...  So we can now decode the code into
           * a stack of character codes. (Clear as mud, right?)
           */
          code = c;

          /*
           * Here we go again with one of those off chances...
           */
          if (code > slot)  { retcode = IE_BAD_FILE_DATA; goto GETOUT; }
          if (code == slot)
            {
              code = oc;
              *sp++ = (Uns8) fc;
            }

          /*
           * Here we scan back along the linked list of prefixes, pushing
           * helpless characters (ie. suffixes) onto the stack as we do so.
           */
          while (code >= newcodes)
            {
              *sp++ = suffix[code];
              code = prefix[code];
            }

          /*
           * Push the last character on the stack, and set up the new
           * prefix and suffix, and if the required slot number is greater
           * than that allowed by the current bit size, increase the bit
           * size.  (NOTE - If we are all full, we *don't* save the new
           * suffix and prefix...  I'm not certain if this is correct...
           * it might be more proper to overwrite the last code...
           */
          *sp++ = (Uns8) code;
          if (slot < top_slot)
            {
              suffix[slot] = (Uns8) (fc = code);
              prefix[slot++] = oc;
              oc = c;
            }
          if (slot>=top_slot &&  curr_size<12)
            {
              top_slot <<= 1;
              ++curr_size;
            }

          /*
           * Now that we've pushed the decoded string (in reverse order)
           * onto the stack, lets pop it off and put it into our decode
           * buffer...  And when the decode buffer is full, write another
           * line...
           */
          while (sp > stack)
            {
              *bufptr++ = *(--sp);
              if (--bufcnt == 0)
                {
                  out_line(gs,linecount++,bi,mode);
                  bufptr = gs->gifbits;
                  bufcnt = gs->nx;
                }
            }
        }
    }
  GETOUT:
  if (prefix) free(prefix);
  return retcode;
}

/*  -> decoder */
/*  <- gifgraph */

static int PASCAL NEAR
gifgraph(char *fname,HANDLE *hmf,int *nx,int *ny,DWORD mode)
{
  int			retcode;
  FILE			*f;
  BITMAPINFO		*bi;
  BITMAPINFOHEADER	*bih;
  struct gifhdr		gs;

  if ( (f=fopen(fname,"r")) == NULL ) return IE_NO_FILE;
  memset(&gs,0,sizeof(gs));
  if ((bi=malloc(sizeof(BITMAPINFOHEADER)+256*sizeof(RGBQUAD)))==NULL)
    {
      fclose(f);
      return IE_MEM_FULL;
    }
  retcode = header(f,&gs,&(bi->bmiColors[0]));
  if (retcode != IE_OK)
    {
      fclose(f);
      free(bi);
      return retcode;
    }
  gs.hpal = make_palette(gs.ncolors,&(bi->bmiColors[0]));

  bih			= &(bi->bmiHeader);
  memset(bih,0,sizeof(BITMAPINFOHEADER));
  bih->biSize		= sizeof(BITMAPINFOHEADER);
  bih->biWidth		= gs.nx;
  bih->biHeight		= gs.ny;
  bih->biPlanes		= 1;
  bih->biBitCount	= (WORD) gs.bpp;
  retcode = init_meta(&gs);
  if (retcode==IE_OK) retcode = decoder(f,&gs,bi,mode);
  finish_meta(&gs,retcode);
  if (gs.hpal)	DeleteObject(gs.hpal);
  free(bi);
  fclose(f);
  *nx = gs.nx;
  *ny = gs.ny;
  *hmf = gs.hmf;
  return retcode;
}

/*  -> gifgraph */
#define read_graphic(fname,hmf,x,y,m)	gifgraph(fname,hmf,x,y,m)
#include "common2.c"
